'use client';

import Link from 'next/link';
import Image from 'next/image';
import { useRouter } from 'next/navigation';
import { useState } from 'react';
import { FaCalculator, FaHome, FaInfoCircle, FaEnvelope, FaSearch } from 'react-icons/fa';
// import AdBanner from '../ads/AdBanner'; // Disabled until AdSense approval
import { calculators } from '@/data/calculators';
import PlukLogo from '@/images/PlukLogo.png';

export default function Header() {
  const [searchQuery, setSearchQuery] = useState('');
  const [showResults, setShowResults] = useState(false);
  const router = useRouter();

  const filteredCalculators = searchQuery.length > 0
    ? calculators.filter(calc =>
        calc.title.toLowerCase().includes(searchQuery.toLowerCase()) ||
        calc.description.toLowerCase().includes(searchQuery.toLowerCase()) ||
        calc.tags.some(tag => tag.toLowerCase().includes(searchQuery.toLowerCase()))
      )
    : [];

  const handleSearch = (e: React.FormEvent) => {
    e.preventDefault();
    if (filteredCalculators.length > 0) {
      router.push(filteredCalculators[0].path);
      setSearchQuery('');
      setShowResults(false);
    }
  };

  return (
    <header className="bg-white shadow-sm sticky top-0 z-50 border-b-4 border-cyan-500">
      <div className="container mx-auto px-4">
        {/* Top Bar */}
        <div className="flex justify-between items-center py-8 gap-4">
          <Link href="/" className="flex-shrink-0 hover:opacity-80 transition-opacity">
            <Image 
              src={PlukLogo} 
              alt="SA PLUK Logo" 
              width={150} 
              height={50}
              priority
              className="h-12 w-auto"
            />
          </Link>

          {/* Search Bar - Desktop */}
          <div className="hidden lg:block flex-1 max-w-md relative">
            <form onSubmit={handleSearch}>
              <div className="relative">
                <input
                  type="text"
                  placeholder="Search calculators..."
                  className="w-full px-4 py-2 pl-10 pr-4 text-sm border border-gray-300 rounded-full focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  value={searchQuery}
                  onChange={(e) => {
                    setSearchQuery(e.target.value);
                    setShowResults(e.target.value.length > 0);
                  }}
                  onBlur={() => setTimeout(() => setShowResults(false), 200)}
                  onFocus={() => setShowResults(searchQuery.length > 0)}
                />
                <FaSearch className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400" />
              </div>
            </form>

            {/* Search Results Dropdown */}
            {showResults && filteredCalculators.length > 0 && (
              <div className="absolute z-50 w-full mt-2 bg-white border border-gray-200 rounded-lg shadow-lg max-h-96 overflow-y-auto">
                {filteredCalculators.slice(0, 8).map((calc) => (
                  <Link
                    key={calc.id}
                    href={calc.path}
                    className="block px-4 py-3 hover:bg-gray-50 border-b border-gray-100 last:border-b-0"
                    onClick={() => {
                      setSearchQuery('');
                      setShowResults(false);
                    }}
                  >
                    <div className="flex items-center">
                      <span className="text-2xl mr-3">{calc.icon}</span>
                      <div>
                        <p className="text-sm font-medium text-gray-900">{calc.title}</p>
                        <p className="text-xs text-gray-500">{calc.category}</p>
                      </div>
                    </div>
                  </Link>
                ))}
              </div>
            )}
          </div>
          
          <nav className="hidden md:flex space-x-6">
            <Link href="/" className="flex items-center text-gray-700 hover:text-blue-600 transition">
              <FaHome className="mr-1" /> Home
            </Link>
            <Link href="/calculators" className="flex items-center text-gray-700 hover:text-blue-600 transition">
              <FaCalculator className="mr-1" /> Calculators
            </Link>
            <Link href="/blog" className="flex items-center text-gray-700 hover:text-blue-600 transition">
              📝 Blog
            </Link>
            <Link href="/about" className="flex items-center text-gray-700 hover:text-blue-600 transition">
              <FaInfoCircle className="mr-1" /> About
            </Link>
            <Link href="/contact" className="flex items-center text-gray-700 hover:text-blue-600 transition">
              <FaEnvelope className="mr-1" /> Contact
            </Link>
          </nav>
          
          <button className="md:hidden text-gray-700">
            <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 6h16M4 12h16m-7 6h7" />
            </svg>
          </button>
        </div>
        
        {/* Mobile Menu (hidden by default) */}
        <div className="md:hidden py-2 border-t">
          <div className="flex flex-col space-y-2">
            <Link href="/" className="px-2 py-1 text-gray-700 hover:bg-gray-100 rounded">Home</Link>
            <Link href="/calculators" className="px-2 py-1 text-gray-700 hover:bg-gray-100 rounded">All Calculators</Link>
            <Link href="/about" className="px-2 py-1 text-gray-700 hover:bg-gray-100 rounded">About Us</Link>
            <Link href="/contact" className="px-2 py-1 text-gray-700 hover:bg-gray-100 rounded">Contact</Link>
          </div>
        </div>
      </div>
      
      {/* Ad Banner - Disabled until AdSense approval */}
      {/* <div className="bg-gray-100 border-t border-b py-2">
        <div className="container mx-auto px-4">
          <AdBanner slot="top-banner" />
        </div>
      </div> */}
    </header>
  );
}
