import Link from 'next/link';
import { calculators } from '@/data/calculators';

type RelatedCalculatorsProps = {
  currentCalculatorId: string;
  category?: string;
  limit?: number;
};

export default function RelatedCalculators({ 
  currentCalculatorId, 
  category,
  limit = 3 
}: RelatedCalculatorsProps) {
  // Get related calculators based on category or just exclude current
  let related = calculators.filter(calc => calc.id !== currentCalculatorId);
  
  // If category is provided, prioritize same category
  if (category) {
    const sameCategory = related.filter(calc => calc.category === category);
    const otherCategory = related.filter(calc => calc.category !== category);
    related = [...sameCategory, ...otherCategory];
  }
  
  // Limit the results
  const displayCalculators = related.slice(0, limit);

  return (
    <div className="bg-gradient-to-r from-blue-50 to-indigo-50 rounded-xl p-8 my-8">
      <h2 className="text-2xl font-bold text-gray-900 mb-6">Related Calculators</h2>
      <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
        {displayCalculators.map((calc) => (
          <Link
            key={calc.id}
            href={calc.path}
            className="bg-white rounded-lg shadow-md hover:shadow-xl transition-shadow duration-300 overflow-hidden border border-gray-200 group"
          >
            <div className="p-6">
              <div className="text-4xl mb-3">{calc.icon}</div>
              <h3 className="text-lg font-semibold text-gray-900 mb-2 group-hover:text-blue-600 transition-colors">
                {calc.title}
              </h3>
              <p className="text-sm text-gray-600 mb-3">{calc.description}</p>
              <div className="flex items-center justify-between">
                <span className="inline-block bg-blue-100 text-blue-800 text-xs px-3 py-1 rounded-full">
                  {calc.category}
                </span>
                <svg 
                  className="w-5 h-5 text-blue-600 group-hover:translate-x-1 transition-transform" 
                  fill="none" 
                  stroke="currentColor" 
                  viewBox="0 0 24 24"
                >
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 5l7 7-7 7" />
                </svg>
              </div>
            </div>
          </Link>
        ))}
      </div>
      
      <div className="text-center mt-6">
        <Link 
          href="/calculators"
          className="inline-block text-blue-600 hover:text-blue-700 font-semibold text-sm"
        >
          View All Calculators →
        </Link>
      </div>
    </div>
  );
}
