'use client';

import { useState, useMemo } from 'react';
import { LeaderboardAd, InContentAd } from '@/components/ads/AdBanner';
import RelatedCalculators from '@/components/calculators/RelatedCalculators';

type Mode = 'add' | 'remove';

export default function VatCalculatorPage() {
  const [mode, setMode] = useState<Mode>('add');
  const [amount, setAmount] = useState<string>('1000');
  const [vatRate, setVatRate] = useState<string>('15');

  const parsedAmount = useMemo(() => parseFloat(amount) || 0, [amount]);
  const parsedRate = useMemo(() => Math.max(0, parseFloat(vatRate) || 0), [vatRate]);

  const { vatAmount, totalIncl, totalExcl } = useMemo(() => {
    const rate = parsedRate / 100;
    if (mode === 'add') {
      const vat = parsedAmount * rate;
      return {
        vatAmount: vat,
        totalIncl: parsedAmount + vat,
        totalExcl: parsedAmount,
      };
    } else {
      const base = parsedAmount / (1 + rate);
      const vat = parsedAmount - base;
      return {
        vatAmount: vat,
        totalIncl: parsedAmount,
        totalExcl: base,
      };
    }
  }, [mode, parsedAmount, parsedRate]);

  const formatCurrency = (n: number) => new Intl.NumberFormat('en-ZA', { style: 'currency', currency: 'ZAR' }).format(n);

  return (
    <div className="space-y-8">
      <div>
        <h1 className="text-3xl font-bold text-gray-900">VAT Calculator</h1>
        <p className="mt-2 text-gray-600">Add or remove VAT quickly. Default VAT rate is 15% (South Africa).</p>
      </div>

      <LeaderboardAd />

      <div className="bg-white rounded-xl shadow p-6">
        <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
          {/* Inputs */}
          <div className="md:col-span-2 space-y-6">
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">Calculation Mode</label>
              <div className="flex gap-3">
                <button
                  type="button"
                  onClick={() => setMode('add')}
                  className={`px-4 py-2 rounded border ${mode === 'add' ? 'bg-blue-600 text-white border-blue-600' : 'bg-white text-gray-700 border-gray-300'}`}
                >
                  Add VAT
                </button>
                <button
                  type="button"
                  onClick={() => setMode('remove')}
                  className={`px-4 py-2 rounded border ${mode === 'remove' ? 'bg-blue-600 text-white border-blue-600' : 'bg-white text-gray-700 border-gray-300'}`}
                >
                  Remove VAT
                </button>
              </div>
            </div>

            <div>
              <label htmlFor="amount" className="block text-sm font-medium text-gray-700 mb-1">Amount</label>
              <div className="relative">
                <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">R</div>
                <input
                  id="amount"
                  type="text"
                  inputMode="decimal"
                  value={amount}
                  onChange={(e) => { const v = e.target.value.replace(/[^0-9.]/g, ''); setAmount(v); }}
                  className="w-full pl-7 pr-3 py-3 border border-gray-300 rounded focus:ring-blue-500 focus:border-blue-500"
                  placeholder="0.00"
                />
              </div>
              <p className="text-xs text-gray-500 mt-1">{mode === 'add' ? 'Amount excluding VAT' : 'Amount including VAT'}</p>
            </div>

            <div>
              <label htmlFor="rate" className="block text-sm font-medium text-gray-700 mb-1">VAT Rate (%)</label>
              <input
                id="rate"
                type="text"
                inputMode="decimal"
                value={vatRate}
                onChange={(e) => { const v = e.target.value.replace(/[^0-9.]/g, ''); setVatRate(v); }}
                className="w-full px-3 py-3 border border-gray-300 rounded focus:ring-blue-500 focus:border-blue-500"
              />
              <p className="text-xs text-gray-500 mt-1">Default is 15% for South Africa. Adjust if needed.</p>
            </div>
          </div>

          {/* Results */}
          <div className="bg-gray-50 rounded p-4 h-full">
            <h2 className="text-lg font-semibold mb-3">Results</h2>
            <div className="space-y-2 text-sm">
              <div className="flex justify-between"><span>VAT Amount:</span><span className="font-medium">{formatCurrency(vatAmount)}</span></div>
              <div className="flex justify-between"><span>Total (Excl. VAT):</span><span className="font-medium">{formatCurrency(totalExcl)}</span></div>
              <div className="flex justify-between"><span>Total (Incl. VAT):</span><span className="font-medium">{formatCurrency(totalIncl)}</span></div>
            </div>
            <div className="mt-4 text-xs text-gray-500">Formulae: Add VAT = amount × (1 + r). Remove VAT = amount ÷ (1 + r). VAT amount = total − base.</div>
          </div>
        </div>
      </div>

      <InContentAd />

      {/* Related Calculators */}
      <RelatedCalculators currentCalculatorId="vat-calculator" category="Tax" />

      {/* Info content for SEO/AdSense */}
      <div className="prose max-w-none">
        <h2>How VAT Works</h2>
        <p>Value-Added Tax (VAT) is a consumption tax added to the price of goods and services. In South Africa, the standard VAT rate is 15%.</p>
        <h3>Examples</h3>
        <ul>
          <li>Excluding to Including: R1,000 at 15% → R1,150 (VAT R150)</li>
          <li>Including to Excluding: R1,150 at 15% → R1,000 (VAT R150)</li>
        </ul>
        <h3>FAQ</h3>
        <p><strong>Can I change the VAT rate?</strong> Yes, adjust the VAT % field for special cases or other countries.</p>
        <p><strong>Is this legal advice?</strong> No. Calculations are for estimation only. Consult a tax professional for compliance.</p>
      </div>
    </div>
  );
}
