'use client';

import { useState, useEffect } from 'react';
import { LeaderboardAd, InContentAd } from '@/components/ads/AdBanner';

type ConversionMode = 'salary-to-hourly' | 'hourly-to-salary';

export default function SalaryConverter() {
  const [mode, setMode] = useState<ConversionMode>('salary-to-hourly');
  const [annualSalary, setAnnualSalary] = useState<string>('480000');
  const [hourlyRate, setHourlyRate] = useState<string>('250');
  const [hoursPerWeek, setHoursPerWeek] = useState<string>('40');
  const [weeksPerYear, setWeeksPerYear] = useState<string>('52');
  const [vacationWeeks, setVacationWeeks] = useState<string>('3');
  
  const [calculatedHourlyRate, setCalculatedHourlyRate] = useState<number>(0);
  const [calculatedAnnualSalary, setCalculatedAnnualSalary] = useState<number>(0);
  const [monthlySalary, setMonthlySalary] = useState<number>(0);
  const [weeklySalary, setWeeklySalary] = useState<number>(0);
  const [dailyRate, setDailyRate] = useState<number>(0);

  useEffect(() => {
    if (mode === 'salary-to-hourly') {
      calculateHourlyFromSalary();
    } else {
      calculateSalaryFromHourly();
    }
  }, [mode, annualSalary, hourlyRate, hoursPerWeek, weeksPerYear, vacationWeeks]);

  const calculateHourlyFromSalary = () => {
    const salary = parseFloat(annualSalary) || 0;
    const hours = parseFloat(hoursPerWeek) || 40;
    const weeks = parseFloat(weeksPerYear) || 52;
    const vacation = parseFloat(vacationWeeks) || 0;
    
    const workingWeeks = weeks - vacation;
    const totalHours = hours * workingWeeks;
    
    if (totalHours > 0) {
      const hourly = salary / totalHours;
      const monthly = salary / 12;
      const weekly = salary / weeks;
      const daily = hourly * 8; // Assuming 8-hour workday
      
      setCalculatedHourlyRate(hourly);
      setMonthlySalary(monthly);
      setWeeklySalary(weekly);
      setDailyRate(daily);
    }
  };

  const calculateSalaryFromHourly = () => {
    const hourly = parseFloat(hourlyRate) || 0;
    const hours = parseFloat(hoursPerWeek) || 40;
    const weeks = parseFloat(weeksPerYear) || 52;
    const vacation = parseFloat(vacationWeeks) || 0;
    
    const workingWeeks = weeks - vacation;
    const annual = hourly * hours * workingWeeks;
    const monthly = annual / 12;
    const weekly = hourly * hours;
    const daily = hourly * 8; // Assuming 8-hour workday
    
    setCalculatedAnnualSalary(annual);
    setMonthlySalary(monthly);
    setWeeklySalary(weekly);
    setDailyRate(daily);
  };

  const formatCurrency = (amount: number): string => {
    return new Intl.NumberFormat('en-ZA', {
      style: 'currency',
      currency: 'ZAR',
      minimumFractionDigits: 2,
      maximumFractionDigits: 2,
    }).format(amount);
  };

  const workingWeeks = parseFloat(weeksPerYear) - parseFloat(vacationWeeks);
  const totalWorkingHours = parseFloat(hoursPerWeek) * workingWeeks;

  return (
    <div className="space-y-8">
      {/* Page Header */}
      <div>
        <h1 className="text-3xl font-bold text-gray-900">Salary to Hourly Converter</h1>
        <p className="mt-2 text-gray-600">
          Convert between annual salary and hourly wage with South African context.
        </p>
      </div>

      <LeaderboardAd />

      {/* Calculator */}
      <div className="bg-white rounded-xl shadow-md p-6">
        {/* Mode Toggle */}
        <div className="flex justify-center mb-6">
          <div className="inline-flex rounded-lg border border-gray-300 p-1">
            <button
              type="button"
              onClick={() => setMode('salary-to-hourly')}
              className={`px-6 py-2 rounded-md text-sm font-medium transition-colors ${
                mode === 'salary-to-hourly'
                  ? 'bg-blue-600 text-white'
                  : 'text-gray-700 hover:bg-gray-100'
              }`}
            >
              Salary → Hourly
            </button>
            <button
              type="button"
              onClick={() => setMode('hourly-to-salary')}
              className={`px-6 py-2 rounded-md text-sm font-medium transition-colors ${
                mode === 'hourly-to-salary'
                  ? 'bg-blue-600 text-white'
                  : 'text-gray-700 hover:bg-gray-100'
              }`}
            >
              Hourly → Salary
            </button>
          </div>
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
          {/* Left Column - Inputs */}
          <div className="lg:col-span-2 space-y-6">
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              {mode === 'salary-to-hourly' ? (
                <div className="md:col-span-2">
                  <label htmlFor="annual-salary" className="block text-sm font-medium text-gray-700 mb-1">
                    Annual Salary (R)
                  </label>
                  <input
                    type="text"
                    id="annual-salary"
                    className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-3 text-sm border-gray-300 rounded-md border"
                    value={annualSalary}
                    onChange={(e) => setAnnualSalary(e.target.value.replace(/[^0-9.]/g, ''))}
                    placeholder="480000"
                  />
                </div>
              ) : (
                <div className="md:col-span-2">
                  <label htmlFor="hourly-rate" className="block text-sm font-medium text-gray-700 mb-1">
                    Hourly Rate (R)
                  </label>
                  <input
                    type="text"
                    id="hourly-rate"
                    className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-3 text-sm border-gray-300 rounded-md border"
                    value={hourlyRate}
                    onChange={(e) => setHourlyRate(e.target.value.replace(/[^0-9.]/g, ''))}
                    placeholder="250"
                  />
                </div>
              )}

              <div>
                <label htmlFor="hours-per-week" className="block text-sm font-medium text-gray-700 mb-1">
                  Hours per Week
                </label>
                <input
                  type="text"
                  id="hours-per-week"
                  className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-2 text-sm border-gray-300 rounded-md border"
                  value={hoursPerWeek}
                  onChange={(e) => setHoursPerWeek(e.target.value.replace(/[^0-9.]/g, ''))}
                  placeholder="40"
                />
              </div>

              <div>
                <label htmlFor="weeks-per-year" className="block text-sm font-medium text-gray-700 mb-1">
                  Weeks per Year
                </label>
                <input
                  type="text"
                  id="weeks-per-year"
                  className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-2 text-sm border-gray-300 rounded-md border"
                  value={weeksPerYear}
                  onChange={(e) => setWeeksPerYear(e.target.value.replace(/[^0-9.]/g, ''))}
                  placeholder="52"
                />
              </div>

              <div className="md:col-span-2">
                <label htmlFor="vacation-weeks" className="block text-sm font-medium text-gray-700 mb-1">
                  Vacation/Leave Weeks per Year
                </label>
                <input
                  type="text"
                  id="vacation-weeks"
                  className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-2 text-sm border-gray-300 rounded-md border"
                  value={vacationWeeks}
                  onChange={(e) => setVacationWeeks(e.target.value.replace(/[^0-9.]/g, ''))}
                  placeholder="3"
                />
                <p className="mt-1 text-xs text-gray-500">SA minimum: 21 days (3 weeks) annual leave</p>
              </div>
            </div>
          </div>

          {/* Right Column - Results */}
          <div className="space-y-4">
            <div className="bg-gradient-to-br from-blue-50 to-indigo-100 rounded-lg p-6">
              <h3 className="text-lg font-semibold text-gray-900 mb-4">Conversion Results</h3>
              
              <div className="space-y-3">
                {mode === 'salary-to-hourly' ? (
                  <>
                    <div>
                      <p className="text-xs text-gray-600">Hourly Rate</p>
                      <p className="text-2xl font-bold text-blue-600">{formatCurrency(calculatedHourlyRate)}/hr</p>
                    </div>
                    <div className="border-t border-indigo-200 pt-3">
                      <p className="text-xs text-gray-600">Annual Salary</p>
                      <p className="text-lg font-semibold text-gray-900">{formatCurrency(parseFloat(annualSalary) || 0)}</p>
                    </div>
                  </>
                ) : (
                  <>
                    <div>
                      <p className="text-xs text-gray-600">Annual Salary</p>
                      <p className="text-2xl font-bold text-blue-600">{formatCurrency(calculatedAnnualSalary)}</p>
                    </div>
                    <div className="border-t border-indigo-200 pt-3">
                      <p className="text-xs text-gray-600">Hourly Rate</p>
                      <p className="text-lg font-semibold text-gray-900">{formatCurrency(parseFloat(hourlyRate) || 0)}/hr</p>
                    </div>
                  </>
                )}

                <div className="border-t border-indigo-200 pt-3">
                  <p className="text-xs text-gray-600">Monthly Salary</p>
                  <p className="text-lg font-semibold text-gray-900">{formatCurrency(monthlySalary)}</p>
                </div>

                <div>
                  <p className="text-xs text-gray-600">Weekly Salary</p>
                  <p className="text-sm font-medium text-gray-900">{formatCurrency(weeklySalary)}</p>
                </div>

                <div>
                  <p className="text-xs text-gray-600">Daily Rate (8 hrs)</p>
                  <p className="text-sm font-medium text-gray-900">{formatCurrency(dailyRate)}</p>
                </div>
              </div>
            </div>

            <div className="bg-gray-50 rounded-lg p-4 text-sm">
              <p className="text-gray-700 mb-2">
                <strong>Working Hours:</strong>
              </p>
              <ul className="space-y-1 text-gray-600">
                <li>• {hoursPerWeek} hours/week</li>
                <li>• {workingWeeks} working weeks/year</li>
                <li>• {totalWorkingHours.toFixed(0)} total hours/year</li>
              </ul>
            </div>
          </div>
        </div>
      </div>

      <InContentAd />

      {/* Information Section */}
      <div className="prose max-w-none">
        <h2>Understanding Salary vs Hourly Pay in South Africa</h2>
        <p>
          In South Africa, employees can be paid either a fixed salary or an hourly rate. Understanding the conversion 
          between these two is important for comparing job offers, negotiating pay, or understanding your true hourly worth.
        </p>

        <h2>How to Use the Salary Converter</h2>
        <ol>
          <li>Choose whether you want to convert from salary to hourly or vice versa</li>
          <li>Enter your annual salary or hourly rate</li>
          <li>Adjust hours per week (standard is 40-45 hours in SA)</li>
          <li>Set weeks per year (usually 52)</li>
          <li>Enter vacation/leave weeks (minimum 21 days or 3 weeks in SA)</li>
          <li>View your converted rate and breakdown</li>
        </ol>

        <h2>South African Working Hours and Leave</h2>
        <p>
          According to the Basic Conditions of Employment Act (BCEA) in South Africa:
        </p>
        <ul>
          <li><strong>Maximum Working Hours:</strong> 45 hours per week (9 hours per day for 5-day week, or 8 hours per day for 6-day week)</li>
          <li><strong>Annual Leave:</strong> Minimum 21 consecutive days per year (or 1 day for every 17 days worked)</li>
          <li><strong>Sick Leave:</strong> 30 days per 3-year cycle (or 1 day for every 26 days worked)</li>
          <li><strong>Public Holidays:</strong> 12 public holidays per year (paid days off)</li>
          <li><strong>Overtime:</strong> Time and a half for weekdays, double time for Sundays and public holidays</li>
        </ul>

        <h2>Salary vs Hourly: Pros and Cons</h2>
        
        <h3>Salaried Employees</h3>
        <p><strong>Advantages:</strong></p>
        <ul>
          <li>Predictable monthly income</li>
          <li>Often includes benefits (medical aid, pension, car allowance)</li>
          <li>Paid leave and sick days</li>
          <li>Job security and career progression</li>
        </ul>
        <p><strong>Disadvantages:</strong></p>
        <ul>
          <li>No overtime pay (unless specified in contract)</li>
          <li>May work longer hours without extra compensation</li>
          <li>Less flexibility in work hours</li>
        </ul>

        <h3>Hourly Employees</h3>
        <p><strong>Advantages:</strong></p>
        <ul>
          <li>Paid for every hour worked</li>
          <li>Overtime opportunities increase earnings</li>
          <li>More flexibility in choosing hours</li>
          <li>Clear compensation for extra work</li>
        </ul>
        <p><strong>Disadvantages:</strong></p>
        <ul>
          <li>Income can vary month to month</li>
          <li>May not receive benefits</li>
          <li>No paid leave (unless specified)</li>
          <li>Less job security</li>
        </ul>

        <h2>Tax Considerations</h2>
        <p>
          Remember that your gross salary (before tax) is different from your net salary (take-home pay). 
          In South Africa, PAYE (Pay As You Earn) tax is deducted from your salary. Other deductions may include:
        </p>
        <ul>
          <li><strong>PAYE Tax:</strong> Progressive tax rates from 18% to 45%</li>
          <li><strong>UIF:</strong> Unemployment Insurance Fund (1% of salary, capped)</li>
          <li><strong>Pension/Provident Fund:</strong> Retirement savings (often 7.5-15%)</li>
          <li><strong>Medical Aid:</strong> Health insurance contributions</li>
        </ul>

        <h2>Freelance and Contract Rates</h2>
        <p>
          If you're a freelancer or contractor, your hourly rate should be higher than an equivalent salaried position because:
        </p>
        <ul>
          <li>You don't receive benefits (medical aid, pension)</li>
          <li>No paid leave or sick days</li>
          <li>You pay your own tax (provisional tax)</li>
          <li>Business expenses (equipment, software, workspace)</li>
          <li>Income uncertainty and gaps between contracts</li>
        </ul>
        <p>
          A common rule of thumb: multiply your desired salary hourly rate by 1.5 to 2 times for freelance work.
        </p>

        <h2>Negotiating Your Salary</h2>
        <p>
          When negotiating a job offer, consider:
        </p>
        <ul>
          <li>Total compensation package (salary + benefits)</li>
          <li>Industry standards for your role and experience</li>
          <li>Cost of living in your area</li>
          <li>Opportunities for growth and bonuses</li>
          <li>Work-life balance and flexibility</li>
        </ul>

        <div className="bg-blue-50 border-l-4 border-blue-400 p-4 my-6">
          <p className="text-sm">
            <strong>Tip:</strong> When comparing job offers, convert everything to the same metric (annual salary or hourly rate) 
            and factor in benefits, leave, and working hours to make an accurate comparison.
          </p>
        </div>
      </div>
    </div>
  );
}
