'use client';

import { useState, useEffect } from 'react';
import { LeaderboardAd, InContentAd } from '@/components/ads/AdBanner';

export default function NetWorthCalculator() {
  // Assets
  const [cash, setCash] = useState<string>('50000');
  const [investments, setInvestments] = useState<string>('150000');
  const [retirement, setRetirement] = useState<string>('200000');
  const [property, setProperty] = useState<string>('1500000');
  const [vehicles, setVehicles] = useState<string>('250000');
  const [otherAssets, setOtherAssets] = useState<string>('50000');

  // Liabilities
  const [homeLoan, setHomeLoan] = useState<string>('800000');
  const [carLoan, setCarLoan] = useState<string>('150000');
  const [creditCard, setCreditCard] = useState<string>('25000');
  const [personalLoan, setPersonalLoan] = useState<string>('50000');
  const [studentLoan, setStudentLoan] = useState<string>('0');
  const [otherDebt, setOtherDebt] = useState<string>('10000');

  const [totalAssets, setTotalAssets] = useState<number>(0);
  const [totalLiabilities, setTotalLiabilities] = useState<number>(0);
  const [netWorth, setNetWorth] = useState<number>(0);
  const [assetBreakdown, setAssetBreakdown] = useState<any[]>([]);
  const [liabilityBreakdown, setLiabilityBreakdown] = useState<any[]>([]);

  useEffect(() => {
    calculateNetWorth();
  }, [cash, investments, retirement, property, vehicles, otherAssets, homeLoan, carLoan, creditCard, personalLoan, studentLoan, otherDebt]);

  const calculateNetWorth = () => {
    const assets = {
      cash: parseFloat(cash) || 0,
      investments: parseFloat(investments) || 0,
      retirement: parseFloat(retirement) || 0,
      property: parseFloat(property) || 0,
      vehicles: parseFloat(vehicles) || 0,
      other: parseFloat(otherAssets) || 0,
    };

    const liabilities = {
      homeLoan: parseFloat(homeLoan) || 0,
      carLoan: parseFloat(carLoan) || 0,
      creditCard: parseFloat(creditCard) || 0,
      personalLoan: parseFloat(personalLoan) || 0,
      studentLoan: parseFloat(studentLoan) || 0,
      other: parseFloat(otherDebt) || 0,
    };

    const totalA = Object.values(assets).reduce((sum, val) => sum + val, 0);
    const totalL = Object.values(liabilities).reduce((sum, val) => sum + val, 0);
    const nw = totalA - totalL;

    setTotalAssets(totalA);
    setTotalLiabilities(totalL);
    setNetWorth(nw);

    // Asset breakdown
    setAssetBreakdown([
      { label: 'Cash & Savings', value: assets.cash, color: 'bg-green-500' },
      { label: 'Investments', value: assets.investments, color: 'bg-blue-500' },
      { label: 'Retirement', value: assets.retirement, color: 'bg-purple-500' },
      { label: 'Property', value: assets.property, color: 'bg-yellow-500' },
      { label: 'Vehicles', value: assets.vehicles, color: 'bg-red-500' },
      { label: 'Other Assets', value: assets.other, color: 'bg-gray-500' },
    ]);

    // Liability breakdown
    setLiabilityBreakdown([
      { label: 'Home Loan', value: liabilities.homeLoan, color: 'bg-red-600' },
      { label: 'Car Loan', value: liabilities.carLoan, color: 'bg-orange-600' },
      { label: 'Credit Card', value: liabilities.creditCard, color: 'bg-yellow-600' },
      { label: 'Personal Loan', value: liabilities.personalLoan, color: 'bg-pink-600' },
      { label: 'Student Loan', value: liabilities.studentLoan, color: 'bg-indigo-600' },
      { label: 'Other Debt', value: liabilities.other, color: 'bg-gray-600' },
    ]);
  };

  const formatCurrency = (amount: number): string => {
    return new Intl.NumberFormat('en-ZA', {
      style: 'currency',
      currency: 'ZAR',
      minimumFractionDigits: 0,
      maximumFractionDigits: 0,
    }).format(amount);
  };

  const getNetWorthStatus = () => {
    if (netWorth >= 5000000) return { text: 'Excellent', color: 'text-green-600' };
    if (netWorth >= 1000000) return { text: 'Very Good', color: 'text-blue-600' };
    if (netWorth >= 500000) return { text: 'Good', color: 'text-indigo-600' };
    if (netWorth >= 100000) return { text: 'Fair', color: 'text-yellow-600' };
    if (netWorth >= 0) return { text: 'Building', color: 'text-orange-600' };
    return { text: 'Negative', color: 'text-red-600' };
  };

  const status = getNetWorthStatus();

  return (
    <div className="space-y-8">
      {/* Page Header */}
      <div>
        <h1 className="text-3xl font-bold text-gray-900">Net Worth Calculator</h1>
        <p className="mt-2 text-gray-600">
          Calculate your total net worth by adding all assets and subtracting all liabilities.
        </p>
      </div>

      <LeaderboardAd />

      {/* Calculator */}
      <div className="bg-white rounded-xl shadow-md p-6">
        <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
          {/* Left Column - Inputs */}
          <div className="lg:col-span-2 space-y-6">
            {/* Assets Section */}
            <div>
              <h3 className="text-lg font-semibold text-green-700 mb-4">Assets (What You Own)</h3>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                  <label htmlFor="cash" className="block text-sm font-medium text-gray-700 mb-1">
                    Cash & Bank Accounts
                  </label>
                  <input
                    type="text"
                    id="cash"
                    className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-2 text-sm border-gray-300 rounded-md border"
                    value={cash}
                    onChange={(e) => setCash(e.target.value.replace(/[^0-9.]/g, ''))}
                    placeholder="50000"
                  />
                </div>

                <div>
                  <label htmlFor="investments" className="block text-sm font-medium text-gray-700 mb-1">
                    Investments (Shares, Unit Trusts)
                  </label>
                  <input
                    type="text"
                    id="investments"
                    className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-2 text-sm border-gray-300 rounded-md border"
                    value={investments}
                    onChange={(e) => setInvestments(e.target.value.replace(/[^0-9.]/g, ''))}
                    placeholder="150000"
                  />
                </div>

                <div>
                  <label htmlFor="retirement" className="block text-sm font-medium text-gray-700 mb-1">
                    Retirement (RA, Pension, Provident)
                  </label>
                  <input
                    type="text"
                    id="retirement"
                    className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-2 text-sm border-gray-300 rounded-md border"
                    value={retirement}
                    onChange={(e) => setRetirement(e.target.value.replace(/[^0-9.]/g, ''))}
                    placeholder="200000"
                  />
                </div>

                <div>
                  <label htmlFor="property" className="block text-sm font-medium text-gray-700 mb-1">
                    Property (Current Market Value)
                  </label>
                  <input
                    type="text"
                    id="property"
                    className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-2 text-sm border-gray-300 rounded-md border"
                    value={property}
                    onChange={(e) => setProperty(e.target.value.replace(/[^0-9.]/g, ''))}
                    placeholder="1500000"
                  />
                </div>

                <div>
                  <label htmlFor="vehicles" className="block text-sm font-medium text-gray-700 mb-1">
                    Vehicles (Current Value)
                  </label>
                  <input
                    type="text"
                    id="vehicles"
                    className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-2 text-sm border-gray-300 rounded-md border"
                    value={vehicles}
                    onChange={(e) => setVehicles(e.target.value.replace(/[^0-9.]/g, ''))}
                    placeholder="250000"
                  />
                </div>

                <div>
                  <label htmlFor="other-assets" className="block text-sm font-medium text-gray-700 mb-1">
                    Other Assets (Jewelry, Art, etc.)
                  </label>
                  <input
                    type="text"
                    id="other-assets"
                    className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-2 text-sm border-gray-300 rounded-md border"
                    value={otherAssets}
                    onChange={(e) => setOtherAssets(e.target.value.replace(/[^0-9.]/g, ''))}
                    placeholder="50000"
                  />
                </div>
              </div>
            </div>

            {/* Liabilities Section */}
            <div className="border-t pt-6">
              <h3 className="text-lg font-semibold text-red-700 mb-4">Liabilities (What You Owe)</h3>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                  <label htmlFor="home-loan" className="block text-sm font-medium text-gray-700 mb-1">
                    Home Loan / Bond Balance
                  </label>
                  <input
                    type="text"
                    id="home-loan"
                    className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-2 text-sm border-gray-300 rounded-md border"
                    value={homeLoan}
                    onChange={(e) => setHomeLoan(e.target.value.replace(/[^0-9.]/g, ''))}
                    placeholder="800000"
                  />
                </div>

                <div>
                  <label htmlFor="car-loan" className="block text-sm font-medium text-gray-700 mb-1">
                    Car Loan Balance
                  </label>
                  <input
                    type="text"
                    id="car-loan"
                    className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-2 text-sm border-gray-300 rounded-md border"
                    value={carLoan}
                    onChange={(e) => setCarLoan(e.target.value.replace(/[^0-9.]/g, ''))}
                    placeholder="150000"
                  />
                </div>

                <div>
                  <label htmlFor="credit-card" className="block text-sm font-medium text-gray-700 mb-1">
                    Credit Card Balance
                  </label>
                  <input
                    type="text"
                    id="credit-card"
                    className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-2 text-sm border-gray-300 rounded-md border"
                    value={creditCard}
                    onChange={(e) => setCreditCard(e.target.value.replace(/[^0-9.]/g, ''))}
                    placeholder="25000"
                  />
                </div>

                <div>
                  <label htmlFor="personal-loan" className="block text-sm font-medium text-gray-700 mb-1">
                    Personal Loan Balance
                  </label>
                  <input
                    type="text"
                    id="personal-loan"
                    className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-2 text-sm border-gray-300 rounded-md border"
                    value={personalLoan}
                    onChange={(e) => setPersonalLoan(e.target.value.replace(/[^0-9.]/g, ''))}
                    placeholder="50000"
                  />
                </div>

                <div>
                  <label htmlFor="student-loan" className="block text-sm font-medium text-gray-700 mb-1">
                    Student Loan / NSFAS Balance
                  </label>
                  <input
                    type="text"
                    id="student-loan"
                    className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-2 text-sm border-gray-300 rounded-md border"
                    value={studentLoan}
                    onChange={(e) => setStudentLoan(e.target.value.replace(/[^0-9.]/g, ''))}
                    placeholder="0"
                  />
                </div>

                <div>
                  <label htmlFor="other-debt" className="block text-sm font-medium text-gray-700 mb-1">
                    Other Debt (Store Cards, etc.)
                  </label>
                  <input
                    type="text"
                    id="other-debt"
                    className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-2 text-sm border-gray-300 rounded-md border"
                    value={otherDebt}
                    onChange={(e) => setOtherDebt(e.target.value.replace(/[^0-9.]/g, ''))}
                    placeholder="10000"
                  />
                </div>
              </div>
            </div>
          </div>

          {/* Right Column - Results */}
          <div className="space-y-4">
            <div className="bg-gradient-to-br from-blue-50 to-indigo-100 rounded-lg p-6">
              <h3 className="text-lg font-semibold text-gray-900 mb-4">Net Worth Summary</h3>
              
              <div className="space-y-3">
                <div>
                  <p className="text-xs text-gray-600">Your Net Worth</p>
                  <p className={`text-3xl font-bold ${status.color}`}>{formatCurrency(netWorth)}</p>
                </div>

                <div className="border-t border-indigo-200 pt-3">
                  <p className="text-xs text-gray-600">Status</p>
                  <p className={`text-lg font-bold ${status.color}`}>{status.text}</p>
                </div>

                <div className="border-t border-indigo-200 pt-3">
                  <p className="text-xs text-gray-600">Total Assets</p>
                  <p className="text-lg font-semibold text-green-600">{formatCurrency(totalAssets)}</p>
                </div>

                <div>
                  <p className="text-xs text-gray-600">Total Liabilities</p>
                  <p className="text-lg font-semibold text-red-600">{formatCurrency(totalLiabilities)}</p>
                </div>

                <div className="border-t border-indigo-200 pt-3">
                  <p className="text-xs text-gray-600">Asset-to-Liability Ratio</p>
                  <p className="text-sm font-medium text-gray-900">
                    {totalLiabilities > 0 ? (totalAssets / totalLiabilities).toFixed(2) : '∞'}:1
                  </p>
                </div>
              </div>
            </div>

            <div className="bg-gray-50 rounded-lg p-4 text-sm">
              <h4 className="font-semibold text-gray-900 mb-2">Net Worth Ranges (SA)</h4>
              <ul className="space-y-1 text-gray-600 text-xs">
                <li>• R5M+: Excellent</li>
                <li>• R1M-R5M: Very Good</li>
                <li>• R500K-R1M: Good</li>
                <li>• R100K-R500K: Fair</li>
                <li>• R0-R100K: Building</li>
                <li>• Negative: In Debt</li>
              </ul>
            </div>
          </div>
        </div>

        {/* Breakdown Charts */}
        <div className="mt-8 grid grid-cols-1 md:grid-cols-2 gap-8">
          {/* Assets Breakdown */}
          <div>
            <h3 className="text-lg font-semibold text-gray-900 mb-4">Assets Breakdown</h3>
            <div className="space-y-2">
              {assetBreakdown.map((item, index) => (
                <div key={index} className="flex items-center justify-between">
                  <div className="flex items-center flex-1">
                    <div className={`w-3 h-3 ${item.color} rounded-full mr-2`}></div>
                    <span className="text-sm text-gray-700">{item.label}</span>
                  </div>
                  <div className="flex items-center">
                    <span className="text-sm font-medium text-gray-900 mr-2">{formatCurrency(item.value)}</span>
                    <span className="text-xs text-gray-500 w-12 text-right">
                      {totalAssets > 0 ? ((item.value / totalAssets) * 100).toFixed(1) : 0}%
                    </span>
                  </div>
                </div>
              ))}
            </div>
          </div>

          {/* Liabilities Breakdown */}
          <div>
            <h3 className="text-lg font-semibold text-gray-900 mb-4">Liabilities Breakdown</h3>
            <div className="space-y-2">
              {liabilityBreakdown.map((item, index) => (
                <div key={index} className="flex items-center justify-between">
                  <div className="flex items-center flex-1">
                    <div className={`w-3 h-3 ${item.color} rounded-full mr-2`}></div>
                    <span className="text-sm text-gray-700">{item.label}</span>
                  </div>
                  <div className="flex items-center">
                    <span className="text-sm font-medium text-gray-900 mr-2">{formatCurrency(item.value)}</span>
                    <span className="text-xs text-gray-500 w-12 text-right">
                      {totalLiabilities > 0 ? ((item.value / totalLiabilities) * 100).toFixed(1) : 0}%
                    </span>
                  </div>
                </div>
              ))}
            </div>
          </div>
        </div>
      </div>

      <InContentAd />

      {/* Information Section */}
      <div className="prose max-w-none">
        <h2>Understanding Net Worth in South Africa</h2>
        <p>
          Your net worth is the total value of everything you own (assets) minus everything you owe (liabilities). 
          It's a snapshot of your financial health and a key indicator of long-term wealth building.
        </p>

        <h2>How to Calculate Net Worth</h2>
        <p>
          The formula is simple: <strong>Net Worth = Total Assets - Total Liabilities</strong>
        </p>
        <p>
          For example, if you have R2.2 million in assets and R1.035 million in liabilities, 
          your net worth is R1.165 million.
        </p>

        <h2>What Counts as Assets?</h2>
        <ul>
          <li><strong>Liquid Assets:</strong> Cash, savings accounts, money market accounts</li>
          <li><strong>Investments:</strong> Shares, unit trusts, ETFs, bonds, TFSA</li>
          <li><strong>Retirement:</strong> Retirement annuity (RA), pension fund, provident fund</li>
          <li><strong>Property:</strong> Home, rental properties, land (current market value)</li>
          <li><strong>Vehicles:</strong> Cars, motorcycles, boats (current resale value)</li>
          <li><strong>Other:</strong> Jewelry, art, collectibles, business equity</li>
        </ul>

        <h2>What Counts as Liabilities?</h2>
        <ul>
          <li><strong>Home Loan:</strong> Outstanding bond balance</li>
          <li><strong>Vehicle Finance:</strong> Car loan balance</li>
          <li><strong>Credit Cards:</strong> Total credit card debt</li>
          <li><strong>Personal Loans:</strong> Bank loans, payday loans</li>
          <li><strong>Student Loans:</strong> NSFAS or private student loan balance</li>
          <li><strong>Other Debt:</strong> Store cards, furniture accounts, medical debt</li>
        </ul>

        <h2>Net Worth Benchmarks by Age (South Africa)</h2>
        <p>
          While everyone's situation is different, here are rough guidelines for net worth by age in South Africa:
        </p>
        <ul>
          <li><strong>Age 25-30:</strong> R50,000 - R200,000 (building phase)</li>
          <li><strong>Age 30-35:</strong> R200,000 - R500,000 (growth phase)</li>
          <li><strong>Age 35-40:</strong> R500,000 - R1,000,000 (accumulation)</li>
          <li><strong>Age 40-50:</strong> R1,000,000 - R3,000,000 (peak earning)</li>
          <li><strong>Age 50-60:</strong> R2,000,000 - R5,000,000 (pre-retirement)</li>
          <li><strong>Age 60+:</strong> R3,000,000+ (retirement)</li>
        </ul>

        <h2>How to Increase Your Net Worth</h2>
        <p>
          There are only two ways to increase net worth: increase assets or decrease liabilities (or both).
        </p>

        <h3>Increase Assets</h3>
        <ul>
          <li><strong>Save More:</strong> Increase your savings rate (aim for 20% of income)</li>
          <li><strong>Invest Wisely:</strong> Put money in growth assets (shares, property, unit trusts)</li>
          <li><strong>Maximize Retirement:</strong> Contribute to RA/pension (tax benefits up to 27.5%)</li>
          <li><strong>Side Income:</strong> Start a side business or freelance work</li>
          <li><strong>Increase Earnings:</strong> Negotiate raises, change jobs, upskill</li>
          <li><strong>Property Investment:</strong> Buy rental property for passive income</li>
        </ul>

        <h3>Decrease Liabilities</h3>
        <ul>
          <li><strong>Pay Off High-Interest Debt:</strong> Focus on credit cards and store cards first</li>
          <li><strong>Avoid New Debt:</strong> Live within your means, avoid lifestyle inflation</li>
          <li><strong>Refinance:</strong> Get better interest rates on existing loans</li>
          <li><strong>Extra Payments:</strong> Pay more than minimum on loans to reduce principal</li>
          <li><strong>Debt Consolidation:</strong> Combine multiple debts into one lower-rate loan</li>
        </ul>

        <h2>Common Net Worth Mistakes</h2>
        <ul>
          <li><strong>Overvaluing Assets:</strong> Use realistic market values, not purchase prices</li>
          <li><strong>Forgetting Liabilities:</strong> Include all debts, even small ones</li>
          <li><strong>Ignoring Depreciation:</strong> Vehicles and electronics lose value over time</li>
          <li><strong>Not Tracking Regularly:</strong> Calculate net worth at least annually</li>
          <li><strong>Comparing to Others:</strong> Focus on your own progress, not others'</li>
        </ul>

        <h2>Net Worth vs Income</h2>
        <p>
          High income doesn't automatically mean high net worth. Many high earners have low net worth due to:
        </p>
        <ul>
          <li>Lifestyle inflation (spending increases with income)</li>
          <li>High debt levels (expensive cars, homes, credit cards)</li>
          <li>Poor investment habits (not saving or investing)</li>
          <li>Lack of financial planning</li>
        </ul>
        <p>
          Conversely, moderate earners can build substantial net worth through disciplined saving, 
          smart investing, and avoiding debt.
        </p>

        <h2>Tracking Your Progress</h2>
        <p>
          Calculate your net worth at least once a year (e.g., on your birthday or New Year's Day). 
          Track the trend over time - it should generally increase as you age. If it's decreasing, 
          review your spending, debt, and investment strategy.
        </p>

        <div className="bg-blue-50 border-l-4 border-blue-400 p-4 my-6">
          <p className="text-sm">
            <strong>Tip:</strong> A negative net worth isn't the end of the world, especially for young people 
            with student loans or those who recently bought property. Focus on the trend - is it improving? 
            That's what matters most.
          </p>
        </div>
      </div>
    </div>
  );
}
