'use client';

import { useState, useEffect } from 'react';
import { PieChart, Pie, Cell, ResponsiveContainer, Tooltip, Legend } from 'recharts';
import { LeaderboardAd, InContentAd } from '@/components/ads/AdBanner';
import RelatedCalculators from '@/components/calculators/RelatedCalculators';

type PaymentFrequency = 'monthly' | 'bi-weekly' | 'weekly';

interface AmortizationEntry {
  paymentNumber: number;
  paymentDate: string;
  payment: number;
  principal: number;
  interest: number;
  remainingBalance: number;
  totalInterest: number;
}

export default function LoanCalculator() {
  // Form inputs
  const [loanAmount, setLoanAmount] = useState<string>('10000');
  const [interestRate, setInterestRate] = useState<string>('5');
  const [loanTerm, setLoanTerm] = useState<string>('5');
  const [paymentFrequency, setPaymentFrequency] = useState<PaymentFrequency>('monthly');
  const [startDate, setStartDate] = useState<string>(new Date().toISOString().split('T')[0]);
  
  // Results
  const [monthlyPayment, setMonthlyPayment] = useState<number>(0);
  const [totalInterest, setTotalInterest] = useState<number>(0);
  const [totalPayment, setTotalPayment] = useState<number>(0);
  const [amortizationSchedule, setAmortizationSchedule] = useState<AmortizationEntry[]>([]);
  const [showAmortization, setShowAmortization] = useState<boolean>(false);
  
  // Calculate loan details when inputs change
  useEffect(() => {
    calculateLoan();
  }, [loanAmount, interestRate, loanTerm, paymentFrequency]);
  
  const calculateLoan = () => {
    const principal = parseFloat(loanAmount);
    const rate = parseFloat(interestRate) / 100;
    const years = parseFloat(loanTerm);
    
    if (isNaN(principal) || isNaN(rate) || isNaN(years) || principal <= 0 || years <= 0) {
      resetResults();
      return;
    }
    
    // Determine payments per year based on frequency
    let paymentsPerYear: number;
    switch (paymentFrequency) {
      case 'monthly':
        paymentsPerYear = 12;
        break;
      case 'bi-weekly':
        paymentsPerYear = 26;
        break;
      case 'weekly':
        paymentsPerYear = 52;
        break;
      default:
        paymentsPerYear = 12;
    }
    
    const totalPayments = years * paymentsPerYear;
    const periodicRate = rate / paymentsPerYear;
    
    // Calculate payment using the loan payment formula
    let payment: number;
    if (periodicRate === 0) {
      payment = principal / totalPayments;
    } else {
      payment = (principal * periodicRate * Math.pow(1 + periodicRate, totalPayments)) / 
                (Math.pow(1 + periodicRate, totalPayments) - 1);
    }
    
    // Generate amortization schedule
    const schedule: AmortizationEntry[] = [];
    let remainingBalance = principal;
    let totalInterestPaid = 0;
    const startDateObj = new Date(startDate);
    
    for (let i = 1; i <= totalPayments; i++) {
      // Calculate payment date
      const paymentDate = new Date(startDateObj);
      if (paymentFrequency === 'monthly') {
        paymentDate.setMonth(paymentDate.getMonth() + (i - 1));
      } else if (paymentFrequency === 'bi-weekly') {
        paymentDate.setDate(paymentDate.getDate() + ((i - 1) * 14));
      } else { // weekly
        paymentDate.setDate(paymentDate.getDate() + ((i - 1) * 7));
      }
      
      const interestPayment = remainingBalance * periodicRate;
      let principalPayment = payment - interestPayment;
      
      // Adjust the last payment if needed to account for floating point errors
      if (i === totalPayments) {
        principalPayment = remainingBalance;
      }
      
      remainingBalance -= principalPayment;
      totalInterestPaid += interestPayment;
      
      // Ensure remaining balance doesn't go below 0 due to floating point errors
      if (remainingBalance < 0.01) remainingBalance = 0;
      
      schedule.push({
        paymentNumber: i,
        paymentDate: paymentDate.toLocaleDateString(),
        payment: payment,
        principal: principalPayment,
        interest: interestPayment,
        remainingBalance: remainingBalance >= 0 ? remainingBalance : 0,
        totalInterest: totalInterestPaid
      });
      
      // Break if loan is paid off
      if (remainingBalance <= 0) break;
    }
    
    // Update state
    setMonthlyPayment(payment);
    setTotalInterest(totalInterestPaid);
    setTotalPayment(principal + totalInterestPaid);
    setAmortizationSchedule(schedule);
  };
  
  const resetResults = () => {
    setMonthlyPayment(0);
    setTotalInterest(0);
    setTotalPayment(0);
    setAmortizationSchedule([]);
  };
  
  const formatCurrency = (amount: number): string => {
    return new Intl.NumberFormat('en-ZA', {
      style: 'currency',
      currency: 'ZAR',
      minimumFractionDigits: 2,
      maximumFractionDigits: 2
    }).format(amount);
  };
  
  const formatPercent = (value: number): string => {
    return `${value.toFixed(2)}%`;
  };
  
  const handleLoanAmountChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const value = e.target.value.replace(/[^0-9.]/g, '');
    setLoanAmount(value);
  };
  
  const handleInterestRateChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const value = e.target.value.replace(/[^0-9.]/g, '');
    setInterestRate(value);
  };
  
  const handleLoanTermChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const value = e.target.value.replace(/[^0-9]/g, '');
    setLoanTerm(value);
  };
  
  const toggleAmortization = () => {
    setShowAmortization(!showAmortization);
  };
  
  return (
    <div className="space-y-8">
      {/* Page Header */}
      <div>
        <h1 className="text-3xl font-bold text-gray-900">Loan Calculator</h1>
        <p className="mt-2 text-gray-600">
          Calculate your monthly payments, total interest, and amortization schedule for any loan.
        </p>
      </div>

      {/* Top Banner Ad */}
      <LeaderboardAd />

      {/* Calculator */}
      <div className="bg-white rounded-xl shadow-md p-6">
        <div className="grid grid-cols-1 md:grid-cols-2 gap-8">
          {/* Left Column - Inputs */}
          <div className="space-y-6">
            <div>
              <label htmlFor="loan-amount" className="block text-sm font-medium text-gray-700 mb-1">
                Loan Amount
              </label>
              <div className="relative rounded-md shadow-sm">
                <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                  <span className="text-gray-500 sm:text-sm">$</span>
                </div>
                <input
                  type="text"
                  id="loan-amount"
                  className="focus:ring-blue-500 focus:border-blue-500 block w-full pl-7 pr-12 py-3 sm:text-sm border-gray-300 rounded-md border"
                  placeholder="0.00"
                  value={loanAmount}
                  onChange={handleLoanAmountChange}
                  inputMode="decimal"
                />
              </div>
            </div>
            
            <div>
              <label htmlFor="interest-rate" className="block text-sm font-medium text-gray-700 mb-1">
                Interest Rate
              </label>
              <div className="relative rounded-md shadow-sm">
                <input
                  type="text"
                  id="interest-rate"
                  className="focus:ring-blue-500 focus:border-blue-500 block w-full pr-12 py-3 sm:text-sm border-gray-300 rounded-md border pl-3"
                  placeholder="5.0"
                  value={interestRate}
                  onChange={handleInterestRateChange}
                  inputMode="decimal"
                />
                <div className="absolute inset-y-0 right-0 pr-3 flex items-center pointer-events-none">
                  <span className="text-gray-500 sm:text-sm" id="price-currency">%</span>
                </div>
              </div>
            </div>
            
            <div>
              <label htmlFor="loan-term" className="block text-sm font-medium text-gray-700 mb-1">
                Loan Term (Years)
              </label>
              <div className="relative rounded-md shadow-sm">
                <input
                  type="text"
                  id="loan-term"
                  className="focus:ring-blue-500 focus:border-blue-500 block w-full pr-12 py-3 sm:text-sm border-gray-300 rounded-md border pl-3"
                  placeholder="5"
                  value={loanTerm}
                  onChange={handleLoanTermChange}
                  inputMode="numeric"
                />
                <div className="absolute inset-y-0 right-0 pr-3 flex items-center pointer-events-none">
                  <span className="text-gray-500 sm:text-sm">years</span>
                </div>
              </div>
            </div>
            
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Payment Frequency
              </label>
              <div className="mt-1">
                <div className="grid grid-cols-3 gap-3">
                  {[
                    { value: 'monthly', label: 'Monthly' },
                    { value: 'bi-weekly', label: 'Bi-Weekly' },
                    { value: 'weekly', label: 'Weekly' },
                  ].map((option) => (
                    <div key={option.value} className="flex items-center">
                      <input
                        id={`frequency-${option.value}`}
                        name="payment-frequency"
                        type="radio"
                        className="focus:ring-blue-500 h-4 w-4 text-blue-600 border-gray-300"
                        checked={paymentFrequency === option.value}
                        onChange={() => setPaymentFrequency(option.value as PaymentFrequency)}
                      />
                      <label htmlFor={`frequency-${option.value}`} className="ml-2 block text-sm text-gray-700">
                        {option.label}
                      </label>
                    </div>
                  ))}
                </div>
              </div>
            </div>
            
            <div>
              <label htmlFor="start-date" className="block text-sm font-medium text-gray-700 mb-1">
                Start Date
              </label>
              <input
                type="date"
                id="start-date"
                className="focus:ring-blue-500 focus:border-blue-500 block w-full py-2 px-3 sm:text-sm border border-gray-300 rounded-md"
                value={startDate}
                onChange={(e) => setStartDate(e.target.value)}
              />
            </div>
          </div>
          
          {/* Right Column - Results */}
          <div className="bg-gray-50 rounded-lg p-6">
            <h2 className="text-lg font-semibold text-gray-900 mb-4">Loan Summary</h2>
            
            {monthlyPayment > 0 ? (
              <div className="space-y-4">
                <div className="flex justify-between">
                  <span className="text-sm font-medium text-gray-500">Payment Amount:</span>
                  <span className="text-sm font-semibold">{formatCurrency(monthlyPayment)}</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-sm font-medium text-gray-500">Payment Frequency:</span>
                  <span className="text-sm font-semibold capitalize">{paymentFrequency}</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-sm font-medium text-gray-500">Total Interest:</span>
                  <span className="text-sm font-semibold">{formatCurrency(totalInterest)}</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-sm font-medium text-gray-500">Total of {amortizationSchedule.length} Payments:</span>
                  <span className="text-sm font-semibold">{formatCurrency(totalPayment)}</span>
                </div>
                
                <div className="pt-4 mt-4 border-t border-gray-200">
                  <div className="h-2 w-full bg-gray-200 rounded-full overflow-hidden">
                    <div 
                      className="h-full bg-blue-600 rounded-full" 
                      style={{ 
                        width: `${(parseFloat(loanAmount) / totalPayment) * 100}%`,
                        maxWidth: '100%'
                      }}
                    ></div>
                  </div>
                  <div className="flex justify-between mt-2 text-xs text-gray-500">
                    <span>Principal: {formatCurrency(parseFloat(loanAmount))}</span>
                    <span>Interest: {formatCurrency(totalInterest)}</span>
                  </div>
                </div>
                
                <button
                  type="button"
                  onClick={toggleAmortization}
                  className="mt-4 w-full bg-white py-2 px-4 border border-gray-300 rounded-md shadow-sm text-sm font-medium text-gray-700 hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
                >
                  {showAmortization ? 'Hide' : 'View'} Amortization Schedule
                </button>
              </div>
            ) : (
              <div className="text-center py-8">
                <svg
                  className="mx-auto h-12 w-12 text-gray-400"
                  fill="none"
                  viewBox="0 0 24 24"
                  stroke="currentColor"
                >
                  <path
                    strokeLinecap="round"
                    strokeLinejoin="round"
                    strokeWidth={1}
                    d="M9 7h6m0 10v-3m-3 3h.01M9 17h.01M9 14h.01M12 14h.01M15 11h.01M12 11h.01M9 11h.01M7 21h10a2 2 0 002-2V5a2 2 0 00-2-2H7a2 2 0 00-2 2v14a2 2 0 002 2z"
                  />
                </svg>
                <h3 className="mt-2 text-sm font-medium text-gray-900">Enter loan details</h3>
                <p className="mt-1 text-sm text-gray-500">Fill in the form to see your payment details.</p>
              </div>
            )}
          </div>
        </div>
      </div>
      
      {/* Amortization Schedule */}
      {showAmortization && amortizationSchedule.length > 0 && (
        <div className="bg-white rounded-xl shadow-md overflow-hidden">
          <div className="px-6 py-4 border-b border-gray-200">
            <h3 className="text-lg font-medium text-gray-900">Amortization Schedule</h3>
            <p className="mt-1 text-sm text-gray-500">
              A complete table of periodic loan payments, showing the amount of principal and interest that comprises each payment.
            </p>
          </div>
          <div className="overflow-x-auto">
            <table className="min-w-full divide-y divide-gray-200">
              <thead className="bg-gray-50">
                <tr>
                  <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Payment #
                  </th>
                  <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Date
                  </th>
                  <th scope="col" className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Payment
                  </th>
                  <th scope="col" className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Principal
                  </th>
                  <th scope="col" className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Interest
                  </th>
                  <th scope="col" className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Total Interest
                  </th>
                  <th scope="col" className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Balance
                  </th>
                </tr>
              </thead>
              <tbody className="bg-white divide-y divide-gray-200">
                {amortizationSchedule.map((entry, index) => (
                  <tr key={entry.paymentNumber} className={index % 2 === 0 ? 'bg-white' : 'bg-gray-50'}>
                    <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                      {entry.paymentNumber}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                      {entry.paymentDate}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-sm text-right text-gray-900 font-medium">
                      {formatCurrency(entry.payment)}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-sm text-right text-gray-900">
                      {formatCurrency(entry.principal)}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-sm text-right text-gray-500">
                      {formatCurrency(entry.interest)}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-sm text-right text-gray-500">
                      {formatCurrency(entry.totalInterest)}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-sm text-right text-gray-900 font-medium">
                      {formatCurrency(entry.remainingBalance)}
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
          <div className="bg-gray-50 px-6 py-3 flex items-center justify-between border-t border-gray-200">
            <div className="flex-1 flex justify-between sm:hidden">
              <button className="relative inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50">
                Previous
              </button>
              <button className="ml-3 relative inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50">
                Next
              </button>
            </div>
            <div className="hidden sm:flex-1 sm:flex sm:items-center sm:justify-between">
              <div>
                <p className="text-sm text-gray-700">
                  Showing <span className="font-medium">1</span> to <span className="font-medium">10</span> of{' '}
                  <span className="font-medium">{amortizationSchedule.length}</span> payments
                </p>
              </div>
              <div>
                <nav className="relative z-0 inline-flex rounded-md shadow-sm -space-x-px" aria-label="Pagination">
                  <button
                    className="relative inline-flex items-center px-2 py-2 rounded-l-md border border-gray-300 bg-white text-sm font-medium text-gray-500 hover:bg-gray-50"
                  >
                    <span className="sr-only">Previous</span>
                    <svg className="h-5 w-5" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor" aria-hidden="true">
                      <path fillRule="evenodd" d="M12.707 5.293a1 1 0 010 1.414L9.414 10l3.293 3.293a1 1 0 01-1.414 1.414l-4-4a1 1 0 010-1.414l4-4a1 1 0 011.414 0z" clipRule="evenodd" />
                    </svg>
                  </button>
                  <button
                    aria-current="page"
                    className="z-10 bg-blue-50 border-blue-500 text-blue-600 relative inline-flex items-center px-4 py-2 border text-sm font-medium"
                  >
                    1
                  </button>
                  <button
                    className="bg-white border-gray-300 text-gray-500 hover:bg-gray-50 relative inline-flex items-center px-4 py-2 border text-sm font-medium"
                  >
                    2
                  </button>
                  <button
                    className="bg-white border-gray-300 text-gray-500 hover:bg-gray-50 relative inline-flex items-center px-4 py-2 border text-sm font-medium"
                  >
                    3
                  </button>
                  <button
                    className="relative inline-flex items-center px-2 py-2 rounded-r-md border border-gray-300 bg-white text-sm font-medium text-gray-500 hover:bg-gray-50"
                  >
                    <span className="sr-only">Next</span>
                    <svg className="h-5 w-5" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor" aria-hidden="true">
                      <path fillRule="evenodd" d="M7.293 14.707a1 1 0 010-1.414L10.586 10 7.293 6.707a1 1 0 011.414-1.414l4 4a1 1 0 010 1.414l-4 4a1 1 0 01-1.414 0z" clipRule="evenodd" />
                    </svg>
                  </button>
                </nav>
              </div>
            </div>
          </div>
          <div className="bg-gray-50 px-6 py-3 flex justify-end">
            <button
              type="button"
              className="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md shadow-sm text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
            >
              <svg className="-ml-1 mr-2 h-5 w-5" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor" aria-hidden="true">
                <path fillRule="evenodd" d="M3 17a1 1 0 011-1h12a1 1 0 110 2H4a1 1 0 01-1-1zm3.293-7.707a1 1 0 011.414 0L9 10.586V3a1 1 0 112 0v7.586l1.293-1.293a1 1 0 111.414 1.414l-3 3a1 1 0 01-1.414 0l-3-3a1 1 0 010-1.414z" clipRule="evenodd" />
              </svg>
              Download as CSV
            </button>
          </div>
        </div>
      )}
      
      {/* In-Content Ad */}
      <InContentAd />

      {/* Payment Breakdown Chart */}
      {totalPayment > 0 && (
        <div className="bg-white rounded-xl shadow-md p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-4">Payment Breakdown</h3>
          <ResponsiveContainer width="100%" height={300}>
            <PieChart>
              <Pie
                data={[
                  { name: 'Principal', value: parseFloat(loanAmount) || 0, fill: '#3b82f6' },
                  { name: 'Interest', value: totalInterest, fill: '#ef4444' },
                ]}
                cx="50%"
                cy="50%"
                labelLine={false}
                label={({ name, percent }: any) => `${name}: ${(percent * 100).toFixed(0)}%`}
                outerRadius={100}
                dataKey="value"
              >
              </Pie>
              <Tooltip formatter={(value: number) => formatCurrency(value)} />
              <Legend />
            </PieChart>
          </ResponsiveContainer>
        </div>
      )}

      {/* Related Calculators */}
      <RelatedCalculators currentCalculatorId="loan-calculator" category="Loans" />

      {/* Information Section */}
      <div className="prose max-w-none">
        <h2>How to Use the Loan Calculator</h2>
        <ol>
          <li><strong>Enter the loan amount</strong> - The total amount you plan to borrow.</li>
          <li><strong>Input the interest rate</strong> - The annual interest rate for your loan.</li>
          <li><strong>Set the loan term</strong> - The number of years you'll take to repay the loan.</li>
          <li><strong>Choose payment frequency</strong> - How often you'll make payments (monthly, bi-weekly, or weekly).</li>
          <li><strong>Select a start date</strong> - When your first payment will be due.</li>
        </ol>
        <p>The calculator will then show you your estimated monthly payment, total interest paid, and the total cost of the loan.</p>

        <h2>Understanding Your Loan</h2>
        <h3>Monthly Payment</h3>
        <p>This is the amount you'll need to pay each period. It includes both principal and interest.</p>
        
        <h3>Total Interest</h3>
        <p>This is the total amount of interest you'll pay over the life of the loan. Making additional principal payments can significantly reduce this amount.</p>
        
        <h3>Total Payment</h3>
        <p>This is the total amount you'll have paid by the end of the loan term, including both principal and interest.</p>
        
        <h3>Amortization Schedule</h3>
        <p>This detailed table shows each payment's breakdown between principal and interest, and how your loan balance decreases over time.</p>

        <h2>Tips for Managing Your Loan</h2>
        <ul>
          <li><strong>Make bi-weekly payments:</strong> Instead of monthly, make half-payments every two weeks. You'll make one extra payment each year, reducing your loan term and total interest.</li>
          <li><strong>Round up your payments:</strong> Even an extra $20-50 per month can significantly reduce your loan term and interest.</li>
          <li><strong>Refinance when rates drop:</strong> If interest rates decrease, consider refinancing to lower your monthly payments or shorten your loan term.</li>
          <li><strong>Make one extra payment each year:</strong> This can reduce a 30-year mortgage to about 22 years.</li>
          <li><strong>Use windfalls wisely:</strong> Apply bonuses, tax refunds, or other unexpected income to your principal.</li>
        </ul>

        <h2>Frequently Asked Questions</h2>
        <div className="space-y-4">
          <div>
            <h3 className="font-semibold">What's the difference between APR and interest rate?</h3>
            <p className="text-gray-700">
              The interest rate is the cost of borrowing the principal loan amount. The APR (Annual Percentage Rate) includes the interest rate plus any additional fees or costs associated with the loan. APR gives you a more complete picture of the loan's cost.
            </p>
          </div>
          
          <div>
            <h3 className="font-semibold">Should I choose a shorter or longer loan term?</h3>
            <p className="text-gray-700">
              A shorter term means higher monthly payments but less interest paid over the life of the loan. A longer term means lower monthly payments but more interest paid overall. Choose based on your monthly budget and how quickly you want to be debt-free.
            </p>
          </div>
          
          <div>
            <h3 className="font-semibold">How does making extra payments affect my loan?</h3>
            <p className="text-gray-700">
              Extra payments are applied directly to your principal, reducing the amount of interest you'll pay over the life of the loan and potentially allowing you to pay off the loan early.
            </p>
          </div>
          
          <div>
            <h3 className="font-semibold">What's an amortization schedule?</h3>
            <p className="text-gray-700">
              An amortization schedule is a complete table of periodic loan payments, showing the amount of principal and interest that comprises each payment until the loan is paid off. Early in the loan, most of each payment goes toward interest rather than principal.
            </p>
          </div>
        </div>
      </div>
      
      {/* Bottom CTA */}
      <div className="bg-blue-50 rounded-lg p-6 text-center">
        <h2 className="text-xl font-bold mb-2">Need Help With Your Loan?</h2>
        <p className="text-gray-700 mb-4 max-w-2xl mx-auto">
          Our loan experts can help you find the best loan options and answer any questions you have about the borrowing process.
        </p>
        <button className="bg-blue-600 hover:bg-blue-700 text-white font-semibold py-2 px-6 rounded-lg transition duration-300 inline-flex items-center">
          <svg className="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M3 5a2 2 0 012-2h3.28a1 1 0 01.948.684l1.498 4.493a1 1 0 01-.502 1.21l-2.257 1.13a11.042 11.042 0 005.516 5.516l1.13-2.257a1 1 0 011.21-.502l4.493 1.498a1 1 0 01.684.949V19a2 2 0 01-2 2h-1C9.716 21 3 14.284 3 6V5z" />
          </svg>
          Speak with a Loan Specialist
        </button>
      </div>
    </div>
  );
}
