'use client';

import { useState, useEffect } from 'react';
import { useSearchParams } from 'next/navigation';
import { LeaderboardAd, InContentAd } from '@/components/ads/AdBanner';
import RelatedCalculators from '@/components/calculators/RelatedCalculators';

// Common currencies with their symbols and names
const CURRENCIES = [
  { code: 'USD', name: 'US Dollar', symbol: '$' },
  { code: 'EUR', name: 'Euro', symbol: '€' },
  { code: 'GBP', name: 'British Pound', symbol: '£' },
  { code: 'JPY', name: 'Japanese Yen', symbol: '¥' },
  { code: 'AUD', name: 'Australian Dollar', symbol: 'A$' },
  { code: 'CAD', name: 'Canadian Dollar', symbol: 'C$' },
  { code: 'CHF', name: 'Swiss Franc', symbol: 'CHF' },
  { code: 'CNY', name: 'Chinese Yuan', symbol: '¥' },
  { code: 'INR', name: 'Indian Rupee', symbol: '₹' },
  { code: 'ZAR', name: 'South African Rand', symbol: 'R' },
  { code: 'NZD', name: 'New Zealand Dollar', symbol: 'NZ$' },
  { code: 'SGD', name: 'Singapore Dollar', symbol: 'S$' },
  { code: 'HKD', name: 'Hong Kong Dollar', symbol: 'HK$' },
  { code: 'SEK', name: 'Swedish Krona', symbol: 'kr' },
  { code: 'NOK', name: 'Norwegian Krone', symbol: 'kr' },
  { code: 'KRW', name: 'South Korean Won', symbol: '₩' },
  { code: 'TRY', name: 'Turkish Lira', symbol: '₺' },
  { code: 'MXN', name: 'Mexican Peso', symbol: 'MX$' },
  { code: 'BRL', name: 'Brazilian Real', symbol: 'R$' },
  { code: 'RUB', name: 'Russian Ruble', symbol: '₽' },
];

type ExchangeRates = {
  [key: string]: number;
};

export default function CurrencyConverter() {
  const searchParams = useSearchParams();
  
  // State for the form
  const [amount, setAmount] = useState<string>('1');
  const [fromCurrency, setFromCurrency] = useState<string>('USD');
  const [toCurrency, setToCurrency] = useState<string>('EUR');
  const [result, setResult] = useState<string>('');
  const [exchangeRates, setExchangeRates] = useState<ExchangeRates>({});
  const [lastUpdated, setLastUpdated] = useState<string>('');
  const [isLoading, setIsLoading] = useState<boolean>(true);
  const [error, setError] = useState<string | null>(null);
  const [showSwap, setShowSwap] = useState<boolean>(true);

  // Get currencies from URL params if they exist
  useEffect(() => {
    const from = searchParams.get('from');
    const to = searchParams.get('to');
    const amt = searchParams.get('amount');
    
    if (from && CURRENCIES.some(c => c.code === from.toUpperCase())) {
      setFromCurrency(from.toUpperCase());
    }
    
    if (to && CURRENCIES.some(c => c.code === to.toUpperCase())) {
      setToCurrency(to.toUpperCase());
    }
    
    if (amt && !isNaN(parseFloat(amt))) {
      setAmount(amt);
    }
  }, [searchParams]);

  // Fetch exchange rates
  useEffect(() => {
    const fetchExchangeRates = async () => {
      setIsLoading(true);
      setError(null);
      
      try {
        // Use exchangerate-api.com free API (no key required for basic usage)
        const response = await fetch('https://api.exchangerate-api.com/v4/latest/USD');
        
        if (!response.ok) {
          throw new Error('Failed to fetch exchange rates');
        }
        
        const data = await response.json();
        
        // Extract rates from the API response
        const rates: ExchangeRates = {
          USD: 1, // Base currency
          ...data.rates
        };
        
        setExchangeRates(rates);
        setLastUpdated(new Date().toLocaleString('en-US', { timeZone: 'UTC' }));
      } catch (err) {
        console.error('Error fetching exchange rates:', err);
        setError('Failed to fetch live exchange rates. Using cached data.');
        
        // Fallback to sample data if API fails
        const sampleRates: ExchangeRates = {
          USD: 1,
          EUR: 0.92,
          GBP: 0.79,
          JPY: 150.25,
          AUD: 1.53,
          CAD: 1.35,
          CHF: 0.88,
          CNY: 7.19,
          INR: 82.96,
          ZAR: 18.92,
          NZD: 1.65,
          SGD: 1.34,
          HKD: 7.83,
          SEK: 10.45,
          NOK: 10.68,
          KRW: 1320.50,
          TRY: 28.75,
          MXN: 17.12,
          BRL: 4.98,
          RUB: 92.50,
        };
        setExchangeRates(sampleRates);
        setLastUpdated(new Date().toLocaleString('en-US', { timeZone: 'UTC' }) + ' (cached)');
      } finally {
        setIsLoading(false);
      }
    };
    
    fetchExchangeRates();
    
    // Refresh rates every 10 minutes (API updates hourly)
    const interval = setInterval(fetchExchangeRates, 10 * 60 * 1000);
    
    return () => clearInterval(interval);
  }, []);

  // Calculate the converted amount whenever inputs change
  useEffect(() => {
    if (!amount || isNaN(parseFloat(amount)) || !exchangeRates[fromCurrency] || !exchangeRates[toCurrency]) {
      setResult('');
      return;
    }
    
    const amountNum = parseFloat(amount);
    const fromRate = exchangeRates[fromCurrency];
    const toRate = exchangeRates[toCurrency];
    
    // Convert to USD first, then to the target currency
    const usdAmount = amountNum / fromRate;
    const convertedAmount = usdAmount * toRate;
    
    // Format the result based on the amount
    let formattedResult: string;
    if (Math.abs(convertedAmount) >= 1000) {
      formattedResult = convertedAmount.toLocaleString('en-US', {
        minimumFractionDigits: 2,
        maximumFractionDigits: 2,
      });
    } else if (Math.abs(convertedAmount) >= 0.01) {
      formattedResult = convertedAmount.toLocaleString('en-US', {
        minimumFractionDigits: 2,
        maximumFractionDigits: 6,
      });
    } else {
      formattedResult = convertedAmount.toExponential(4);
    }
    
    setResult(formattedResult);
  }, [amount, fromCurrency, toCurrency, exchangeRates]);

  // Handle input changes
  const handleAmountChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const value = e.target.value;
    // Allow numbers, decimal point, and empty string
    if (value === '' || /^\d*\.?\d*$/.test(value)) {
      setAmount(value);
    }
  };

  // Swap from and to currencies
  const handleSwapCurrencies = () => {
    setFromCurrency(toCurrency);
    setToCurrency(fromCurrency);
  };

  // Format currency for display
  const formatCurrency = (amount: string, currencyCode: string) => {
    if (!amount) return '';
    
    const currency = CURRENCIES.find(c => c.code === currencyCode);
    const symbol = currency?.symbol || currencyCode;
    
    return `${symbol} ${amount}`;
  };

  return (
    <div className="space-y-8">
      {/* Page Header */}
      <div>
        <h1 className="text-3xl font-bold text-gray-900">Currency Converter</h1>
        <p className="mt-2 text-gray-600">
          Convert between different currencies with live exchange rates.
        </p>
      </div>

      {/* Top Banner Ad */}
      <LeaderboardAd />

      {/* Converter Tool */}
      <div className="bg-white rounded-xl shadow-md p-6">
        {isLoading && (
          <div className="text-center py-4">
            <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-500 mx-auto"></div>
            <p className="mt-2 text-gray-600">Loading exchange rates...</p>
          </div>
        )}
        
        {error && (
          <div className="bg-yellow-50 border-l-4 border-yellow-400 p-4 mb-6">
            <div className="flex">
              <div className="flex-shrink-0">
                <svg className="h-5 w-5 text-yellow-400" viewBox="0 0 20 20" fill="currentColor">
                  <path fillRule="evenodd" d="M8.257 3.099c.765-1.36 2.722-1.36 3.486 0l5.58 9.92c.75 1.334-.213 2.98-1.742 2.98H4.42c-1.53 0-2.493-1.646-1.743-2.98l5.58-9.92zM11 13a1 1 0 11-2 0 1 1 0 012 0zm-1-8a1 1 0 00-1 1v3a1 1 0 002 0V6a1 1 0 00-1-1z" clipRule="evenodd" />
                </svg>
              </div>
              <div className="ml-3">
                <p className="text-sm text-yellow-700">
                  {error} Some features may be limited.
                </p>
              </div>
            </div>
          </div>
        )}
        
        {!isLoading && (
          <>
            <div className="space-y-4">
              {/* Amount Input */}
              <div>
                <label htmlFor="amount" className="block text-sm font-medium text-gray-700 mb-1">
                  Amount
                </label>
                <div className="relative rounded-md shadow-sm">
                  <input
                    type="text"
                    id="amount"
                    className="focus:ring-blue-500 focus:border-blue-500 block w-full pl-3 pr-12 py-3 sm:text-sm border-gray-300 rounded-md border"
                    placeholder="0.00"
                    value={amount}
                    onChange={handleAmountChange}
                    inputMode="decimal"
                  />
                </div>
              </div>
              
              {/* From Currency */}
              <div>
                <label htmlFor="from-currency" className="block text-sm font-medium text-gray-700 mb-1">
                  From
                </label>
                <div className="mt-1 relative rounded-md shadow-sm">
                  <select
                    id="from-currency"
                    className="focus:ring-blue-500 focus:border-blue-500 h-full py-0 pl-3 pr-8 border-transparent bg-transparent text-gray-700 sm:text-sm rounded-md border border-gray-300 w-full py-3"
                    value={fromCurrency}
                    onChange={(e) => setFromCurrency(e.target.value)}
                  >
                    {CURRENCIES.map((currency) => (
                      <option key={`from-${currency.code}`} value={currency.code}>
                        {currency.code} - {currency.name}
                      </option>
                    ))}
                  </select>
                </div>
              </div>
              
              {/* Swap Button */}
              <div className="flex justify-center py-2">
                <button
                  type="button"
                  onClick={handleSwapCurrencies}
                  className="inline-flex items-center p-2 border border-gray-300 shadow-sm text-sm leading-4 font-medium rounded-full text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
                  title="Swap currencies"
                >
                  <svg className="h-5 w-5 text-gray-500" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M8 7h12m0 0l-4-4m4 4l-4 4m0 6H4m0 0l4 4m-4-4l4-4" />
                  </svg>
                </button>
              </div>
              
              {/* To Currency */}
              <div>
                <label htmlFor="to-currency" className="block text-sm font-medium text-gray-700 mb-1">
                  To
                </label>
                <div className="mt-1 relative rounded-md shadow-sm">
                  <select
                    id="to-currency"
                    className="focus:ring-blue-500 focus:border-blue-500 h-full py-0 pl-3 pr-8 border-transparent bg-transparent text-gray-700 sm:text-sm rounded-md border border-gray-300 w-full py-3"
                    value={toCurrency}
                    onChange={(e) => setToCurrency(e.target.value)}
                  >
                    {CURRENCIES.map((currency) => (
                      <option key={`to-${currency.code}`} value={currency.code}>
                        {currency.code} - {currency.name}
                      </option>
                    ))}
                  </select>
                </div>
              </div>
            </div>
            
            {/* Result */}
            <div className="mt-8 bg-blue-50 rounded-lg p-6">
              <h3 className="text-sm font-medium text-gray-500">Result</h3>
              <div className="mt-1 flex items-baseline">
                <span className="text-3xl font-semibold text-gray-900">
                  {formatCurrency(result, toCurrency)}
                </span>
                {fromCurrency !== toCurrency && (
                  <span className="ml-2 text-sm text-gray-500">
                    {formatCurrency('1', fromCurrency)} = {formatCurrency((exchangeRates[toCurrency] / exchangeRates[fromCurrency]).toFixed(6), toCurrency)}
                  </span>
                )}
              </div>
              {fromCurrency !== toCurrency && (
                <div className="mt-2 text-sm text-gray-500">
                  {formatCurrency('1', toCurrency)} = {formatCurrency((exchangeRates[fromCurrency] / exchangeRates[toCurrency]).toFixed(6), fromCurrency)}
                </div>
              )}
              {lastUpdated && (
                <div className="mt-2 text-xs text-gray-400">
                  Exchange rates last updated: {lastUpdated}
                </div>
              )}
            </div>
          </>
        )}
      </div>

      {/* In-Content Ad */}
      <InContentAd />

      {/* Related Calculators */}
      <RelatedCalculators currentCalculatorId="currency-converter" category="Currency" />

      {/* Information Section */}
      <div className="prose max-w-none">
        <h2>How to Use the Currency Converter</h2>
        <ol>
          <li>Enter the amount you want to convert in the "Amount" field.</li>
          <li>Select the currency you're converting from in the "From" dropdown.</li>
          <li>Select the currency you want to convert to in the "To" dropdown.</li>
          <li>The converted amount will be displayed automatically.</li>
          <li>Click the swap button (↕️) to quickly switch between the "From" and "To" currencies.</li>
        </ol>

        <h2>About Exchange Rates</h2>
        <p>
          Exchange rates are the rates at which one currency can be exchanged for another. 
          They are determined by the foreign exchange market (Forex), where currencies are traded 24 hours a day, five days a week.
        </p>
        <p>
          This converter uses live exchange rates from a reliable API source, updated regularly throughout the day. 
          However, the rates shown are for informational purposes only and may differ from actual market rates at the time of your transaction. 
          For large transactions, it's always best to check with your bank or financial institution for the most accurate and up-to-date rates.
        </p>

        <h2>Popular Currency Conversions</h2>
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4 my-4">
          {[
            { from: 'USD', to: 'EUR' },
            { from: 'EUR', to: 'GBP' },
            { from: 'USD', to: 'JPY' },
            { from: 'GBP', to: 'USD' },
            { from: 'USD', to: 'CAD' },
            { from: 'AUD', to: 'USD' },
          ].map((pair, index) => {
            const fromRate = exchangeRates[pair.from] || 1;
            const toRate = exchangeRates[pair.to] || 1;
            const rate = (toRate / fromRate).toFixed(4);
            
            return (
              <div key={index} className="bg-white p-4 rounded-lg shadow border border-gray-200">
                <div className="text-sm text-gray-600">1 {pair.from} = {rate} {pair.to}</div>
                <div className="text-xs text-gray-500 mt-1">
                  {pair.from}/{pair.to}
                </div>
              </div>
            );
          })}
        </div>

        <h2>Tips for Currency Conversion</h2>
        <ul>
          <li>Check for hidden fees when exchanging currency, especially at airports or hotels where rates are often less favorable.</li>
          <li>Consider using a credit card with no foreign transaction fees when traveling abroad.</li>
          <li>Be aware of the mid-market rate (the real exchange rate) to understand if you're getting a good deal.</li>
          <li>For large transfers, compare rates from different providers as they can vary significantly.</li>
          <li>Keep an eye on currency trends if you have flexibility in when you exchange money.</li>
        </ul>

        <h2>Currency Conversion FAQ</h2>
        <div className="space-y-4">
          <div>
            <h3 className="font-semibold">How often are the exchange rates updated?</h3>
            <p className="text-gray-700">
              Exchange rates are fetched from a live API and automatically refresh every 10 minutes. The API itself updates hourly with real market data. Please note that these are indicative rates and may differ from the rates offered by banks or currency exchange services.
            </p>
          </div>
          <div>
            <h3 className="font-semibold">Why do exchange rates change?</h3>
            <p className="text-gray-700">
              Exchange rates fluctuate constantly due to changes in supply and demand for different currencies. Factors include interest rates, inflation, political stability, and economic performance.
            </p>
          </div>
          <div>
            <h3 className="font-semibold">Is there a fee for using this currency converter?</h3>
            <p className="text-gray-700">
              No, our currency converter is completely free to use. We don't charge any fees or commissions for the conversion calculations.
            </p>
          </div>
        </div>
      </div>
    </div>
  );
}
