'use client';

import { useState, useEffect } from 'react';
import { LeaderboardAd, InContentAd } from '@/components/ads/AdBanner';

export default function CreditCardInterestCalculator() {
  const [balance, setBalance] = useState<string>('15000');
  const [interestRate, setInterestRate] = useState<string>('22.5');
  const [monthlyPayment, setMonthlyPayment] = useState<string>('500');
  const [paymentStrategy, setPaymentStrategy] = useState<'minimum' | 'fixed'>('fixed');
  
  const [monthsToPayoff, setMonthsToPayoff] = useState<number>(0);
  const [totalInterest, setTotalInterest] = useState<number>(0);
  const [totalPaid, setTotalPaid] = useState<number>(0);
  const [paymentSchedule, setPaymentSchedule] = useState<any[]>([]);

  useEffect(() => {
    calculatePayoff();
  }, [balance, interestRate, monthlyPayment, paymentStrategy]);

  const calculatePayoff = () => {
    const principal = parseFloat(balance) || 0;
    const rate = parseFloat(interestRate) / 100 / 12 || 0;
    let payment = parseFloat(monthlyPayment) || 0;

    if (principal <= 0 || rate < 0) {
      resetResults();
      return;
    }

    // If payment is less than interest, it will never be paid off
    const monthlyInterest = principal * rate;
    if (payment <= monthlyInterest) {
      resetResults();
      return;
    }

    let remainingBalance = principal;
    let totalInterestPaid = 0;
    let months = 0;
    const schedule = [];
    const maxMonths = 600; // 50 years max

    while (remainingBalance > 0.01 && months < maxMonths) {
      months++;
      
      // Calculate minimum payment if strategy is minimum (3% of balance or R200, whichever is higher)
      if (paymentStrategy === 'minimum') {
        const minimumPayment = Math.max(remainingBalance * 0.03, 200);
        payment = Math.min(minimumPayment, remainingBalance + (remainingBalance * rate));
      }

      const interestCharge = remainingBalance * rate;
      const principalPayment = Math.min(payment - interestCharge, remainingBalance);
      
      remainingBalance -= principalPayment;
      totalInterestPaid += interestCharge;

      if (months <= 12 || months % 6 === 0 || remainingBalance <= 0.01) {
        schedule.push({
          month: months,
          payment: payment,
          principal: principalPayment,
          interest: interestCharge,
          balance: Math.max(0, remainingBalance),
          totalInterest: totalInterestPaid
        });
      }
    }

    setMonthsToPayoff(months);
    setTotalInterest(totalInterestPaid);
    setTotalPaid(principal + totalInterestPaid);
    setPaymentSchedule(schedule);
  };

  const resetResults = () => {
    setMonthsToPayoff(0);
    setTotalInterest(0);
    setTotalPaid(0);
    setPaymentSchedule([]);
  };

  const formatCurrency = (amount: number): string => {
    return new Intl.NumberFormat('en-ZA', {
      style: 'currency',
      currency: 'ZAR',
      minimumFractionDigits: 2,
      maximumFractionDigits: 2,
    }).format(amount);
  };

  const yearsToPayoff = monthsToPayoff / 12;
  const interestPercentage = parseFloat(balance) > 0 ? (totalInterest / parseFloat(balance)) * 100 : 0;

  return (
    <div className="space-y-8">
      {/* Page Header */}
      <div>
        <h1 className="text-3xl font-bold text-gray-900">Credit Card Interest Calculator</h1>
        <p className="mt-2 text-gray-600">
          Calculate how much interest you'll pay on your credit card balance and how long it will take to pay off.
        </p>
      </div>

      <LeaderboardAd />

      {/* Calculator */}
      <div className="bg-white rounded-xl shadow-md p-6">
        <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
          {/* Left Column - Inputs */}
          <div className="lg:col-span-2 space-y-6">
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <div className="md:col-span-2">
                <label htmlFor="balance" className="block text-sm font-medium text-gray-700 mb-1">
                  Current Credit Card Balance (R)
                </label>
                <input
                  type="text"
                  id="balance"
                  className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-3 text-sm border-gray-300 rounded-md border"
                  value={balance}
                  onChange={(e) => setBalance(e.target.value.replace(/[^0-9.]/g, ''))}
                  placeholder="15000"
                />
              </div>

              <div>
                <label htmlFor="interest-rate" className="block text-sm font-medium text-gray-700 mb-1">
                  Annual Interest Rate (%)
                </label>
                <input
                  type="text"
                  id="interest-rate"
                  className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-2 text-sm border-gray-300 rounded-md border"
                  value={interestRate}
                  onChange={(e) => setInterestRate(e.target.value.replace(/[^0-9.]/g, ''))}
                  placeholder="22.5"
                />
                <p className="mt-1 text-xs text-gray-500">SA average: 18-25%</p>
              </div>

              <div>
                <label htmlFor="monthly-payment" className="block text-sm font-medium text-gray-700 mb-1">
                  Monthly Payment (R)
                </label>
                <input
                  type="text"
                  id="monthly-payment"
                  className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-2 text-sm border-gray-300 rounded-md border"
                  value={monthlyPayment}
                  onChange={(e) => setMonthlyPayment(e.target.value.replace(/[^0-9.]/g, ''))}
                  placeholder="500"
                  disabled={paymentStrategy === 'minimum'}
                />
              </div>

              <div className="md:col-span-2">
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Payment Strategy
                </label>
                <div className="flex gap-4">
                  <label className="flex items-center">
                    <input
                      type="radio"
                      name="strategy"
                      value="fixed"
                      checked={paymentStrategy === 'fixed'}
                      onChange={() => setPaymentStrategy('fixed')}
                      className="mr-2"
                    />
                    <span className="text-sm text-gray-700">Fixed Payment</span>
                  </label>
                  <label className="flex items-center">
                    <input
                      type="radio"
                      name="strategy"
                      value="minimum"
                      checked={paymentStrategy === 'minimum'}
                      onChange={() => setPaymentStrategy('minimum')}
                      className="mr-2"
                    />
                    <span className="text-sm text-gray-700">Minimum Payment Only (3% or R200)</span>
                  </label>
                </div>
              </div>
            </div>
          </div>

          {/* Right Column - Results */}
          <div className="space-y-4">
            <div className="bg-gradient-to-br from-red-50 to-orange-100 rounded-lg p-6">
              <h3 className="text-lg font-semibold text-gray-900 mb-4">Payoff Summary</h3>
              
              <div className="space-y-3">
                <div>
                  <p className="text-xs text-gray-600">Time to Pay Off</p>
                  <p className="text-2xl font-bold text-red-600">
                    {monthsToPayoff > 0 ? `${Math.floor(yearsToPayoff)} yrs ${monthsToPayoff % 12} mos` : 'N/A'}
                  </p>
                </div>

                <div className="border-t border-orange-200 pt-3">
                  <p className="text-xs text-gray-600">Total Interest Paid</p>
                  <p className="text-xl font-bold text-orange-600">{formatCurrency(totalInterest)}</p>
                </div>

                <div>
                  <p className="text-xs text-gray-600">Total Amount Paid</p>
                  <p className="text-lg font-semibold text-gray-900">{formatCurrency(totalPaid)}</p>
                </div>

                <div className="border-t border-orange-200 pt-3">
                  <p className="text-xs text-gray-600">Original Balance</p>
                  <p className="text-sm font-medium text-gray-900">{formatCurrency(parseFloat(balance) || 0)}</p>
                </div>

                <div>
                  <p className="text-xs text-gray-600">Interest as % of Balance</p>
                  <p className="text-sm font-semibold text-red-600">{interestPercentage.toFixed(1)}%</p>
                </div>
              </div>
            </div>

            <div className="bg-yellow-50 border-l-4 border-yellow-400 p-4 text-sm">
              <p className="text-yellow-800">
                <strong>Warning:</strong> Paying only the minimum will take {monthsToPayoff > 0 ? Math.ceil(yearsToPayoff) : 0} years 
                and cost you {formatCurrency(totalInterest)} in interest!
              </p>
            </div>
          </div>
        </div>

        {/* Payment Schedule */}
        {paymentSchedule.length > 0 && (
          <div className="mt-8">
            <h3 className="text-lg font-semibold text-gray-900 mb-4">Payment Schedule</h3>
            <div className="overflow-x-auto">
              <table className="min-w-full divide-y divide-gray-200">
                <thead className="bg-gray-50">
                  <tr>
                    <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Month</th>
                    <th className="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase">Payment</th>
                    <th className="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase">Principal</th>
                    <th className="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase">Interest</th>
                    <th className="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase">Balance</th>
                    <th className="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase">Total Interest</th>
                  </tr>
                </thead>
                <tbody className="bg-white divide-y divide-gray-200">
                  {paymentSchedule.map((row, index) => (
                    <tr key={index} className={index % 2 === 0 ? 'bg-white' : 'bg-gray-50'}>
                      <td className="px-4 py-3 text-sm text-gray-900">{row.month}</td>
                      <td className="px-4 py-3 text-sm text-right text-gray-900">{formatCurrency(row.payment)}</td>
                      <td className="px-4 py-3 text-sm text-right text-blue-600">{formatCurrency(row.principal)}</td>
                      <td className="px-4 py-3 text-sm text-right text-red-600">{formatCurrency(row.interest)}</td>
                      <td className="px-4 py-3 text-sm text-right font-medium text-gray-900">{formatCurrency(row.balance)}</td>
                      <td className="px-4 py-3 text-sm text-right text-orange-600">{formatCurrency(row.totalInterest)}</td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
          </div>
        )}
      </div>

      <InContentAd />

      {/* Information Section */}
      <div className="prose max-w-none">
        <h2>Understanding Credit Card Interest in South Africa</h2>
        <p>
          Credit cards in South Africa typically charge interest rates between 18% and 25% per annum, making them one of 
          the most expensive forms of debt. Understanding how this interest compounds can help you avoid the debt trap.
        </p>

        <h2>How Credit Card Interest Works</h2>
        <p>
          Credit card interest is calculated daily and compounded monthly. This means:
        </p>
        <ul>
          <li>Interest is charged on your outstanding balance each day</li>
          <li>The daily rate is your annual rate divided by 365</li>
          <li>Each month, interest is added to your balance</li>
          <li>Next month, you pay interest on the previous interest (compound interest)</li>
        </ul>

        <h2>The Minimum Payment Trap</h2>
        <p>
          South African credit card providers typically require a minimum payment of 3% of the balance or R200 
          (whichever is higher). While this seems manageable, it's designed to keep you in debt:
        </p>
        <ul>
          <li><strong>Extended Payoff:</strong> Paying only the minimum can take 10-30 years to clear the debt</li>
          <li><strong>Massive Interest:</strong> You'll pay 2-3 times the original amount in interest</li>
          <li><strong>Shrinking Payments:</strong> As balance decreases, so does the minimum payment, extending the term</li>
          <li><strong>Compound Effect:</strong> Interest on interest makes the debt grow exponentially</li>
        </ul>

        <h2>Example: R15,000 at 22.5% Interest</h2>
        <p>
          Let's compare payment strategies:
        </p>
        <ul>
          <li><strong>Minimum Payment Only:</strong> Takes ~15 years, costs ~R18,000 in interest</li>
          <li><strong>R500/month Fixed:</strong> Takes ~3.5 years, costs ~R5,500 in interest</li>
          <li><strong>R1,000/month Fixed:</strong> Takes ~1.5 years, costs ~R2,200 in interest</li>
        </ul>

        <h2>Strategies to Pay Off Credit Card Debt</h2>
        
        <h3>1. Pay More Than the Minimum</h3>
        <p>
          Even R100 extra per month can save thousands in interest and years of payments.
        </p>

        <h3>2. Debt Avalanche Method</h3>
        <p>
          If you have multiple credit cards:
        </p>
        <ul>
          <li>Pay minimum on all cards</li>
          <li>Put extra money toward the highest interest rate card</li>
          <li>Once paid off, move to the next highest rate</li>
          <li>Saves the most money in interest</li>
        </ul>

        <h3>3. Debt Snowball Method</h3>
        <p>
          Alternative approach for motivation:
        </p>
        <ul>
          <li>Pay minimum on all cards</li>
          <li>Put extra money toward the smallest balance</li>
          <li>Quick wins provide psychological boost</li>
          <li>May cost slightly more in interest but easier to stick with</li>
        </ul>

        <h3>4. Balance Transfer</h3>
        <p>
          Some SA banks offer 0% balance transfer promotions:
        </p>
        <ul>
          <li>Transfer high-interest debt to 0% card</li>
          <li>Pay off during promotional period (usually 6-12 months)</li>
          <li>Watch for transfer fees (usually 2-3%)</li>
          <li>Don't use the new card for purchases</li>
        </ul>

        <h3>5. Debt Consolidation Loan</h3>
        <p>
          Take a personal loan at lower interest to pay off credit cards:
        </p>
        <ul>
          <li>Personal loan rates: 12-18% vs credit card 18-25%</li>
          <li>Fixed payment and term</li>
          <li>Single monthly payment</li>
          <li>Close credit cards to avoid re-accumulating debt</li>
        </ul>

        <h2>National Credit Act (NCA) Protections</h2>
        <p>
          South African consumers have rights under the NCA:
        </p>
        <ul>
          <li><strong>Interest Rate Cap:</strong> Maximum rate is repo rate + 21% (currently ~29%)</li>
          <li><strong>Affordability Assessment:</strong> Lenders must ensure you can afford the credit</li>
          <li><strong>Debt Review:</strong> Legal process to restructure unaffordable debt</li>
          <li><strong>Clear Disclosure:</strong> All fees and charges must be disclosed upfront</li>
        </ul>

        <h2>Avoiding Credit Card Debt</h2>
        <ul>
          <li><strong>Pay in Full:</strong> Always pay the full balance to avoid interest</li>
          <li><strong>Budget First:</strong> Only charge what you can afford to pay off</li>
          <li><strong>Emergency Fund:</strong> Build savings to avoid using credit for emergencies</li>
          <li><strong>Track Spending:</strong> Monitor credit card usage closely</li>
          <li><strong>Limit Cards:</strong> Fewer cards = less temptation and easier management</li>
        </ul>

        <h2>When to Seek Help</h2>
        <p>
          If you're struggling with credit card debt, consider:
        </p>
        <ul>
          <li><strong>Debt Counseling:</strong> Free advice from registered debt counselors</li>
          <li><strong>Debt Review:</strong> Legal debt restructuring under NCA</li>
          <li><strong>Financial Advisor:</strong> Professional help with budgeting and debt strategy</li>
          <li><strong>Bank Negotiation:</strong> Some banks will negotiate lower rates or payment plans</li>
        </ul>

        <div className="bg-red-50 border-l-4 border-red-400 p-4 my-6">
          <p className="text-sm">
            <strong>Warning:</strong> Credit card debt can spiral out of control quickly due to high interest rates 
            and compound interest. If you're only making minimum payments, you're in a debt trap. Take action now - 
            increase your payments, consolidate, or seek professional help.
          </p>
        </div>
      </div>
    </div>
  );
}
