'use client';

import { useState, useEffect } from 'react';
import { LeaderboardAd, InContentAd } from '@/components/ads/AdBanner';

export default function BreakEvenCalculator() {
  const [fixedCosts, setFixedCosts] = useState<string>('50000');
  const [variableCostPerUnit, setVariableCostPerUnit] = useState<string>('25');
  const [pricePerUnit, setPricePerUnit] = useState<string>('50');
  const [targetProfit, setTargetProfit] = useState<string>('0');
  
  const [breakEvenUnits, setBreakEvenUnits] = useState<number>(0);
  const [breakEvenRevenue, setBreakEvenRevenue] = useState<number>(0);
  const [contributionMargin, setContributionMargin] = useState<number>(0);
  const [contributionMarginRatio, setContributionMarginRatio] = useState<number>(0);
  const [unitsForTargetProfit, setUnitsForTargetProfit] = useState<number>(0);
  const [revenueForTargetProfit, setRevenueForTargetProfit] = useState<number>(0);
  const [profitAnalysis, setProfitAnalysis] = useState<any[]>([]);

  useEffect(() => {
    calculateBreakEven();
  }, [fixedCosts, variableCostPerUnit, pricePerUnit, targetProfit]);

  const calculateBreakEven = () => {
    const fixed = parseFloat(fixedCosts) || 0;
    const variableCost = parseFloat(variableCostPerUnit) || 0;
    const price = parseFloat(pricePerUnit) || 0;
    const profit = parseFloat(targetProfit) || 0;

    if (price <= variableCost || price <= 0) {
      resetResults();
      return;
    }

    // Contribution margin per unit
    const cm = price - variableCost;
    const cmRatio = (cm / price) * 100;

    // Break-even point in units
    const beUnits = fixed / cm;
    const beRevenue = beUnits * price;

    // Units needed for target profit
    const profitUnits = (fixed + profit) / cm;
    const profitRevenue = profitUnits * price;

    setContributionMargin(cm);
    setContributionMarginRatio(cmRatio);
    setBreakEvenUnits(beUnits);
    setBreakEvenRevenue(beRevenue);
    setUnitsForTargetProfit(profitUnits);
    setRevenueForTargetProfit(profitRevenue);

    // Generate profit analysis for different unit levels
    const analysis = [];
    const maxUnits = Math.ceil(beUnits * 2.5);
    const step = Math.ceil(maxUnits / 10);

    for (let units = 0; units <= maxUnits; units += step) {
      const revenue = units * price;
      const totalVariableCost = units * variableCost;
      const totalCost = fixed + totalVariableCost;
      const profitLoss = revenue - totalCost;

      analysis.push({
        units,
        revenue,
        fixedCost: fixed,
        variableCost: totalVariableCost,
        totalCost,
        profit: profitLoss,
        profitMargin: revenue > 0 ? (profitLoss / revenue) * 100 : 0
      });
    }

    setProfitAnalysis(analysis);
  };

  const resetResults = () => {
    setBreakEvenUnits(0);
    setBreakEvenRevenue(0);
    setContributionMargin(0);
    setContributionMarginRatio(0);
    setUnitsForTargetProfit(0);
    setRevenueForTargetProfit(0);
    setProfitAnalysis([]);
  };

  const formatCurrency = (amount: number): string => {
    return new Intl.NumberFormat('en-ZA', {
      style: 'currency',
      currency: 'ZAR',
      minimumFractionDigits: 0,
      maximumFractionDigits: 0,
    }).format(amount);
  };

  const formatNumber = (num: number): string => {
    return new Intl.NumberFormat('en-ZA', {
      minimumFractionDigits: 0,
      maximumFractionDigits: 0,
    }).format(num);
  };

  return (
    <div className="space-y-8">
      {/* Page Header */}
      <div>
        <h1 className="text-3xl font-bold text-gray-900">Break-Even Calculator</h1>
        <p className="mt-2 text-gray-600">
          Calculate the break-even point for your business or product in South Africa.
        </p>
      </div>

      <LeaderboardAd />

      {/* Calculator */}
      <div className="bg-white rounded-xl shadow-md p-6">
        <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
          {/* Left Column - Inputs */}
          <div className="lg:col-span-2 space-y-6">
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <div className="md:col-span-2">
                <label htmlFor="fixed-costs" className="block text-sm font-medium text-gray-700 mb-1">
                  Fixed Costs per Month (R)
                </label>
                <input
                  type="text"
                  id="fixed-costs"
                  className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-3 text-sm border-gray-300 rounded-md border"
                  value={fixedCosts}
                  onChange={(e) => setFixedCosts(e.target.value.replace(/[^0-9.]/g, ''))}
                  placeholder="50000"
                />
                <p className="mt-1 text-xs text-gray-500">Rent, salaries, insurance, etc.</p>
              </div>

              <div>
                <label htmlFor="variable-cost" className="block text-sm font-medium text-gray-700 mb-1">
                  Variable Cost per Unit (R)
                </label>
                <input
                  type="text"
                  id="variable-cost"
                  className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-2 text-sm border-gray-300 rounded-md border"
                  value={variableCostPerUnit}
                  onChange={(e) => setVariableCostPerUnit(e.target.value.replace(/[^0-9.]/g, ''))}
                  placeholder="25"
                />
                <p className="mt-1 text-xs text-gray-500">Materials, labor per unit</p>
              </div>

              <div>
                <label htmlFor="price-per-unit" className="block text-sm font-medium text-gray-700 mb-1">
                  Selling Price per Unit (R)
                </label>
                <input
                  type="text"
                  id="price-per-unit"
                  className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-2 text-sm border-gray-300 rounded-md border"
                  value={pricePerUnit}
                  onChange={(e) => setPricePerUnit(e.target.value.replace(/[^0-9.]/g, ''))}
                  placeholder="50"
                />
              </div>

              <div className="md:col-span-2">
                <label htmlFor="target-profit" className="block text-sm font-medium text-gray-700 mb-1">
                  Target Profit per Month (R) - Optional
                </label>
                <input
                  type="text"
                  id="target-profit"
                  className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-2 text-sm border-gray-300 rounded-md border"
                  value={targetProfit}
                  onChange={(e) => setTargetProfit(e.target.value.replace(/[^0-9.]/g, ''))}
                  placeholder="0"
                />
              </div>
            </div>
          </div>

          {/* Right Column - Results */}
          <div className="space-y-4">
            <div className="bg-gradient-to-br from-purple-50 to-indigo-100 rounded-lg p-6">
              <h3 className="text-lg font-semibold text-gray-900 mb-4">Break-Even Analysis</h3>
              
              <div className="space-y-3">
                <div>
                  <p className="text-xs text-gray-600">Break-Even Units</p>
                  <p className="text-2xl font-bold text-purple-600">{formatNumber(breakEvenUnits)}</p>
                </div>

                <div className="border-t border-indigo-200 pt-3">
                  <p className="text-xs text-gray-600">Break-Even Revenue</p>
                  <p className="text-xl font-bold text-indigo-600">{formatCurrency(breakEvenRevenue)}</p>
                </div>

                <div className="border-t border-indigo-200 pt-3">
                  <p className="text-xs text-gray-600">Contribution Margin</p>
                  <p className="text-lg font-semibold text-gray-900">{formatCurrency(contributionMargin)}/unit</p>
                </div>

                <div>
                  <p className="text-xs text-gray-600">Contribution Margin %</p>
                  <p className="text-sm font-medium text-gray-900">{contributionMarginRatio.toFixed(1)}%</p>
                </div>

                {parseFloat(targetProfit) > 0 && (
                  <>
                    <div className="border-t border-indigo-200 pt-3">
                      <p className="text-xs text-gray-600">Units for Target Profit</p>
                      <p className="text-lg font-bold text-green-600">{formatNumber(unitsForTargetProfit)}</p>
                    </div>

                    <div>
                      <p className="text-xs text-gray-600">Revenue for Target Profit</p>
                      <p className="text-sm font-semibold text-green-600">{formatCurrency(revenueForTargetProfit)}</p>
                    </div>
                  </>
                )}
              </div>
            </div>

            <div className="bg-gray-50 rounded-lg p-4 text-sm">
              <h4 className="font-semibold text-gray-900 mb-2">Key Metrics</h4>
              <ul className="space-y-1 text-gray-600 text-xs">
                <li>• Fixed Costs: {formatCurrency(parseFloat(fixedCosts) || 0)}</li>
                <li>• Variable Cost/Unit: {formatCurrency(parseFloat(variableCostPerUnit) || 0)}</li>
                <li>• Price/Unit: {formatCurrency(parseFloat(pricePerUnit) || 0)}</li>
                <li>• Margin/Unit: {formatCurrency(contributionMargin)}</li>
              </ul>
            </div>
          </div>
        </div>

        {/* Profit Analysis Table */}
        {profitAnalysis.length > 0 && (
          <div className="mt-8">
            <h3 className="text-lg font-semibold text-gray-900 mb-4">Profit Analysis at Different Volumes</h3>
            <div className="overflow-x-auto">
              <table className="min-w-full divide-y divide-gray-200">
                <thead className="bg-gray-50">
                  <tr>
                    <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Units Sold</th>
                    <th className="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase">Revenue</th>
                    <th className="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase">Fixed Costs</th>
                    <th className="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase">Variable Costs</th>
                    <th className="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase">Total Costs</th>
                    <th className="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase">Profit/Loss</th>
                    <th className="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase">Margin %</th>
                  </tr>
                </thead>
                <tbody className="bg-white divide-y divide-gray-200">
                  {profitAnalysis.map((row, index) => (
                    <tr key={index} className={index % 2 === 0 ? 'bg-white' : 'bg-gray-50'}>
                      <td className="px-4 py-3 text-sm text-gray-900">{formatNumber(row.units)}</td>
                      <td className="px-4 py-3 text-sm text-right text-gray-900">{formatCurrency(row.revenue)}</td>
                      <td className="px-4 py-3 text-sm text-right text-gray-600">{formatCurrency(row.fixedCost)}</td>
                      <td className="px-4 py-3 text-sm text-right text-orange-600">{formatCurrency(row.variableCost)}</td>
                      <td className="px-4 py-3 text-sm text-right font-medium text-gray-900">{formatCurrency(row.totalCost)}</td>
                      <td className={`px-4 py-3 text-sm text-right font-bold ${row.profit >= 0 ? 'text-green-600' : 'text-red-600'}`}>
                        {formatCurrency(row.profit)}
                      </td>
                      <td className="px-4 py-3 text-sm text-right text-gray-900">{row.profitMargin.toFixed(1)}%</td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
          </div>
        )}
      </div>

      <InContentAd />

      {/* Information Section */}
      <div className="prose max-w-none">
        <h2>Understanding Break-Even Analysis for South African Businesses</h2>
        <p>
          Break-even analysis is a critical tool for any business in South Africa. It tells you exactly how many units 
          you need to sell to cover all your costs and start making a profit. This is essential for pricing decisions, 
          business planning, and financial forecasting.
        </p>

        <h2>What is the Break-Even Point?</h2>
        <p>
          The break-even point is where your total revenue equals your total costs. At this point:
        </p>
        <ul>
          <li>You're not making a profit</li>
          <li>You're not making a loss</li>
          <li>All fixed and variable costs are covered</li>
          <li>Any sales beyond this point generate profit</li>
        </ul>

        <h2>Break-Even Formula</h2>
        <p>
          <strong>Break-Even Units = Fixed Costs ÷ (Price per Unit - Variable Cost per Unit)</strong>
        </p>
        <p>
          The denominator (Price - Variable Cost) is called the <strong>Contribution Margin</strong> - 
          the amount each unit contributes toward covering fixed costs and generating profit.
        </p>

        <h2>Fixed vs Variable Costs</h2>
        
        <h3>Fixed Costs (Don't change with production)</h3>
        <ul>
          <li><strong>Rent:</strong> Office, shop, or factory space</li>
          <li><strong>Salaries:</strong> Permanent staff wages</li>
          <li><strong>Insurance:</strong> Business, liability, property insurance</li>
          <li><strong>Utilities:</strong> Basic electricity, water (fixed portion)</li>
          <li><strong>Licenses:</strong> Business licenses, permits</li>
          <li><strong>Equipment Leases:</strong> Monthly machinery or vehicle leases</li>
          <li><strong>Marketing:</strong> Fixed advertising contracts</li>
        </ul>

        <h3>Variable Costs (Change with production)</h3>
        <ul>
          <li><strong>Raw Materials:</strong> Ingredients, components, supplies</li>
          <li><strong>Direct Labor:</strong> Hourly workers, piece-rate workers</li>
          <li><strong>Packaging:</strong> Boxes, labels, wrapping</li>
          <li><strong>Shipping:</strong> Delivery costs per unit</li>
          <li><strong>Sales Commissions:</strong> Percentage of sales</li>
          <li><strong>Transaction Fees:</strong> Card processing fees</li>
        </ul>

        <h2>How to Use Break-Even Analysis</h2>
        
        <h3>1. Pricing Decisions</h3>
        <p>
          If your break-even point is too high (unrealistic sales volume), you may need to:
        </p>
        <ul>
          <li>Increase your selling price</li>
          <li>Reduce variable costs (cheaper materials, more efficient production)</li>
          <li>Reduce fixed costs (cheaper rent, fewer staff)</li>
        </ul>

        <h3>2. Business Viability</h3>
        <p>
          Before starting a business, calculate if you can realistically sell enough units to break even. 
          If the market can't support that volume, the business may not be viable.
        </p>

        <h3>3. Profit Planning</h3>
        <p>
          Use the target profit feature to determine how many units you need to sell to achieve your profit goals.
        </p>

        <h3>4. Risk Assessment</h3>
        <p>
          A lower break-even point means less risk. If sales drop, you can still cover costs with fewer sales.
        </p>

        <h2>Example: SA Coffee Shop</h2>
        <p>
          Let's say you're opening a coffee shop in Johannesburg:
        </p>
        <ul>
          <li><strong>Fixed Costs:</strong> R50,000/month (rent R20k, salaries R25k, utilities R5k)</li>
          <li><strong>Variable Cost per Coffee:</strong> R12 (beans R5, milk R3, cup R2, labor R2)</li>
          <li><strong>Selling Price:</strong> R35 per coffee</li>
        </ul>
        <p>
          <strong>Break-Even Calculation:</strong><br />
          Contribution Margin = R35 - R12 = R23<br />
          Break-Even Units = R50,000 ÷ R23 = 2,174 coffees/month<br />
          That's about 72 coffees per day (30-day month)
        </p>

        <h2>Improving Your Break-Even Point</h2>
        
        <h3>Reduce Fixed Costs</h3>
        <ul>
          <li>Negotiate lower rent or move to cheaper location</li>
          <li>Use part-time staff instead of full-time</li>
          <li>Share office space or equipment</li>
          <li>Outsource non-core functions</li>
        </ul>

        <h3>Reduce Variable Costs</h3>
        <ul>
          <li>Negotiate bulk discounts with suppliers</li>
          <li>Improve production efficiency</li>
          <li>Reduce waste and spoilage</li>
          <li>Find cheaper alternatives without sacrificing quality</li>
        </ul>

        <h3>Increase Selling Price</h3>
        <ul>
          <li>Add value to justify higher prices</li>
          <li>Improve branding and positioning</li>
          <li>Target less price-sensitive customers</li>
          <li>Bundle products or services</li>
        </ul>

        <h2>Limitations of Break-Even Analysis</h2>
        <ul>
          <li><strong>Assumes Constant Costs:</strong> In reality, costs may change with volume</li>
          <li><strong>Assumes All Units Sell:</strong> Doesn't account for inventory or waste</li>
          <li><strong>Single Product Focus:</strong> Complex with multiple products</li>
          <li><strong>Ignores Competition:</strong> Market conditions may limit pricing</li>
          <li><strong>Static Analysis:</strong> Doesn't account for seasonal variations</li>
        </ul>

        <h2>Break-Even for Service Businesses</h2>
        <p>
          For service businesses (consulting, salons, etc.), use billable hours as "units":
        </p>
        <ul>
          <li><strong>Fixed Costs:</strong> Rent, admin salaries, software subscriptions</li>
          <li><strong>Variable Costs:</strong> Hourly contractor rates, materials per job</li>
          <li><strong>Price per Unit:</strong> Hourly billing rate</li>
        </ul>

        <h2>Tax Considerations in South Africa</h2>
        <p>
          Remember to factor in:
        </p>
        <ul>
          <li><strong>VAT:</strong> If registered, you collect 15% VAT (doesn't affect break-even directly)</li>
          <li><strong>Income Tax:</strong> Profit is taxable (28% for companies, progressive for individuals)</li>
          <li><strong>PAYE:</strong> Employee taxes (included in salary costs)</li>
          <li><strong>UIF:</strong> Unemployment insurance (1% of salaries)</li>
        </ul>

        <div className="bg-blue-50 border-l-4 border-blue-400 p-4 my-6">
          <p className="text-sm">
            <strong>Business Tip:</strong> Review your break-even point quarterly. As costs change (rent increases, 
            supplier price changes), your break-even point shifts. Regular analysis helps you stay profitable and 
            make informed pricing decisions.
          </p>
        </div>
      </div>
    </div>
  );
}
